/*****************************************************************************/
/*                                                                           */
/*   VECTREX HP3000 Software Emulator              Copyright K.Wilkins 1996  */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*   Title      : Vectrex Cartridge ROM Emulation function module            */
/*                                                                           */
/*   File Name  : CART_ROM.C                                                 */
/*                                                                           */
/*   Author     : Keith Wilkins                                              */
/*                                                                           */
/*   Version    : 1.00                                                       */
/*                                                                           */
/*   Desciption : This file contains the functions that emulate the Vectrex  */
/*                Cartridge ROM. The ROM write is for patching and debugging */
/*                                                                           */
/*   Functions  : fCartridgeROMLoad()   - Init & Load the System RAM         */
/*                fCartridgeROMSave()   - Save the System RAM area to disk   */
/*                fCartridgeROMRead()   - Read a RAM location                */
/*                fCartridgeROMWrite()  - Write to a RAM location            */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*   Revision History:                                                       */
/*                                                                           */
/*   Version    Date    Who  Description of changes                          */
/*   -------    ----    ---  ----------------------                          */
/*                                                                           */
/*    0.01    14/03/93  K.W  Creation of empty file and descriptions         */
/*    1.00    05/08/96  K.W  Public release of DVE V1.0                      */
/*                                                                           */
/*****************************************************************************/

#ifdef _NO_DEBUG_INFORMATION_

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <standard.h>

#include "vdebug.h"
#include "emu6809.h"
#include "vectrex.h"
#include "cart_rom.h"
#include "dis6809.h" // dis_init_state
T_INT tFirstTime=T_TRUE;

#endif //#ifdef _NO_DEBUG_INFORMATION_

#ifndef _NO_DEBUG_INFORMATION_

/*****************************************************************************/
/*                                                                           */
/*   Function    : fCartridgeROMLoad()                                       */
/*                                                                           */
/*   Author      : K.W                                                       */
/*                                                                           */
/*   Description : This function will open and load a binary file into the   */
/*                 32Kb Cartridge ROM space. Before loading all addresses    */
/*                 will be written to CART_ROM_FILL_VALUE. The file is       */
/*                 opened before the ROM space is cleared as if the load     */
/*                 fails we havent destroyed the current ROM Image.          */
/*                                                                           */
/*   Accepts     : char * Pointer to a file name, any length.                */
/*                                                                           */
/*   Returns     : BOOLEAN T_TRUE/T_FALSE depending on load success.             */
/*                                                                           */
/*****************************************************************************/

T_INT fCartridgeROMLoad(T_PUCHAR pcROMFileName)
{
 T_INT   bRetVal=T_TRUE;
 T_INT   bTmp;
 FILE   *pfCartridgeROM=NULL;
 T_ULONG  tLoop;
 T_UCHAR  tData;

 /* Only on the 1st execution do we declare the ROM Space */
 if(tFirstTime==T_TRUE)
 {
  tFirstTime=T_FALSE;
  if(f6809DeclareAddress(CART_ROM_BASE,CART_ROM_SIZE,fCartridgeROMRead,fCartridgeROMWrite)==T_FALSE)
  {
   fDebug(VERROR,("fCartridgeROMLoad() ROM Space delaration failed"));
   return(T_FALSE);
  }
 }
 /* Enable Cartridge ROM writes, and save current stats */
 bTmp=bGblCartridgeROMWriteEnable;
 bGblCartridgeROMWriteEnable=T_TRUE;
 /* Now clear out the Cartridge ROM area */
 for(tLoop=CART_ROM_BASE;tLoop<=CART_ROM_TOP;tLoop++)
 {
  f6809AddressWrite((T_UWORD)tLoop,CART_ROM_FILL_VALUE);
 }
 fDebug(MED,("fCartridgeROMLoad() Cartridge ROM Space Cleared"));
 /* Try and open the ROM image file */
 if(pcROMFileName!=NULL && strcmp(pcROMFileName,"")!=0)
 {
  T_PCHAR current_path=NULL;
  current_path=get_current_path();
  if (GblBinDir!=NULL)
   change_directory(GblBinDir);
  if (f_file_exist(pcROMFileName)==T_FALSE)
  {
   if (current_path!=NULL)
   {
    change_directory(current_path);
    free(current_path);
    current_path=NULL;
   }
  }
  if ((pfCartridgeROM=fopen(pcROMFileName,"rb"))==NULL)
  {
   fDebug(VERROR,("fCartridgeROMLoad() Couldnt open file \"%s\"",pcROMFileName));
   bRetVal=T_FALSE;
  }
  else
  {
   fDebug(MED,("fCartridgeROMLoad() \"%s\" opened O.K",pcROMFileName));
   tLoop=CART_ROM_BASE;
   while(tLoop<=CART_ROM_TOP && fread(&tData,sizeof(T_UCHAR),1,pfCartridgeROM)==1)
   {
    f6809AddressWrite((T_UWORD)tLoop++,tData);
   }
   fDebug(MED,("fCartridgeROMLoad() Loaded %d Bytes",tLoop-CART_ROM_BASE));
   fclose(pfCartridgeROM);
   if (current_path!=NULL)
   {
    change_directory(current_path);
    free(current_path);
    current_path=NULL;
   }
  }
 }
 /* Restore the ROM writes flag */
 dis_init_state=T_FALSE;
 bGblCartridgeROMWriteEnable=bTmp;
 if (strncmp(atSystemImage+17,"STAR HAWK",strlen("STAR HAWK"))==0)
  sthwkptch=T_TRUE;
 else
  sthwkptch=T_FALSE;
 /* All finished */
 return(bRetVal);
}

/*****************************************************************************/
/*                                                                           */
/*   Function    : fCartridgeROMSave()                                       */
/*                                                                           */
/*   Author      : K.W                                                       */
/*                                                                           */
/*   Description : Open a file on disk, write the contents of the cartridge  */
/*                 to disk and close the file                                */
/*                                                                           */
/*   Accepts     : char * Pointer to a file name, any length.                */
/*                                                                           */
/*   Returns     : T_INTEAN T_TRUE/T_FALSE depending on load success.             */
/*                                                                           */
/*****************************************************************************/

T_INT fCartridgeROMSave(T_PUCHAR pcROMFileName)
{
 FILE    *pfCartridgeROM=NULL;
 T_UWORD   tLoop;
 T_UCHAR   tData;
 /* Try and open the ROM image file */
 T_PCHAR current_path=NULL;
 current_path=get_current_path();
 if (GblBinDir!=NULL)
  change_directory(GblBinDir);
 if (f_file_exist(pcROMFileName)==T_FALSE)
 {
  if (current_path!=NULL)
  {
   change_directory(current_path);
   free(current_path);
   current_path=NULL;
  }
 }
 if((pfCartridgeROM=fopen(pcROMFileName,"wb"))==NULL)
 {
  fDebug(VERROR,("fCartridgeROMSave() Couldnt open file %s",pcROMFileName));
  return(T_FALSE);
 }
 else
 {
  fDebug(MED,("fCartridgeROMSave() %s opened O.K",pcROMFileName));
 }
 /* Now Save CART_ROM_SIZE bytes */
 tLoop=CART_ROM_BASE;
 while(tLoop<CART_ROM_BASE+CART_ROM_SIZE)
 {
  tData=fCartridgeROMRead(tLoop++);
  if (fwrite(&tData,sizeof(T_UCHAR),1,pfCartridgeROM)==0)
  {
   fclose(pfCartridgeROM);
   if (current_path!=NULL)
   {
    change_directory(current_path);
    free(current_path);
    current_path=NULL;
   }
   fDebug(VERROR,("fCartridgeROMSave() File write error, aborting save"));
   return(T_FALSE);
  }
 }
 fDebug(MED,("fCartridgeROMSave() Saved %d Bytes",tLoop-CART_ROM_BASE));
 fclose(pfCartridgeROM);
 if (current_path!=NULL)
 {
  change_directory(current_path);
  free(current_path);
  current_path=NULL;
 }
 /* All finished */
 return(T_TRUE);
}

/*****************************************************************************/
/*                                                                           */
/*   Function    : fCartridgeROMRead()                                       */
/*                                                                           */
/*   Author      : K.W                                                       */
/*                                                                           */
/*   Description : Read a value from the ROM image, apply the address mask   */
/*                 to the address before reading to ensure were not out of   */
/*                 bounds.                                                   */
/*                                                                           */
/*   Accepts     : WORD Address pointer (16bit)                              */
/*                                                                           */
/*   Returns     : BYTE Data value                                           */
/*                                                                           */
/*****************************************************************************/

T_UCHAR fCartridgeROMRead(T_UWORD tAddress)
{
 fDebug(LOW,("fCartridgeROMRead"));
 return(atSystemImage[tAddress]);
}

/*****************************************************************************/
/*                                                                           */
/*   Function    : fCartridgeROMWrite()                                      */
/*                                                                           */
/*   Author      : K.W                                                       */
/*                                                                           */
/*   Description : Write a byte to the image array, the address is masked    */
/*                 before the write to ensure we're within the bounds of the */
/*                 array. The write is only performed if the ROM write       */
/*                 enable flag is T_TRUE.                                      */
/*                                                                           */
/*   Accepts     : WORD Address and a BYTE of Data                           */
/*                                                                           */
/*   Returns     : BYTE of Data                                              */
/*                                                                           */
/*****************************************************************************/

T_UCHAR fCartridgeROMWrite(T_UWORD tAddress,T_UCHAR tData)
{
 if(bGblCartridgeROMWriteEnable==T_FALSE)
 {
  fDebug(VERROR,("fCartridgeROMWrite() Attempt to Write #$%02x to ROM $%04x",tData,tAddress));
 }
 else
 {
  atSystemImage[tAddress]=tData;
 }
 return(tData);
}

#endif //#ifndef _NO_DEBUG_INFORMATION_
