//is_upper = true setzen...
/*****************************************************************************/
/*                                                                           */
/*   VECTREX HP3000 Software Emulator              Copyright K.Wilkins 1996  */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*   Title      : This file contains all of the Debug Monitor command funcs  */
/*                                                                           */
/*   File Name  : MONITOR.C                                                  */
/*                                                                           */
/*   Author     : Keith Wilkins                                              */
/*                                                                           */
/*   Version    : 1.00                                                       */
/*                                                                           */
/*   Desciption : The Vectrex debug monitor commands are all function driven */
/*                and this file conatins all of the command functions that   */
/*                are called by fDebugMonitor() in debug.c.                  */
/*                                                                           */
/*                fDebugMonitor()     - The debug monitor kernel caller      */
/*                fDebugBreakRun()    - Run with breakpoints enabled         */
/*                fDebugDissemble()   - Dissasemble 6809 code                */
/*                fDebugDump()        - Hex to to monitor window             */
/*                fDebugExit()        - Exit from the debug monitor          */
/*                fDebugFind()        - Find a hex string                    */
/*                fDebugHelp()        - Print Monitor help listing           */
/*                fDebugRun()         - Run with breakpoints disabled        */
/*                fDebugSequence()    - Run N steps of the system sequencer  */
/*                fDebugStep()        - Run N single steps of the CPU        */
/*                fDebugTicks()       - Display system tick counts           */
/*                fDebugQuit()        - Quit from the emulator               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*   Revision History:                                                       */
/*                                                                           */
/*   Version    Date    Who  Description of changes                          */
/*   -------    ----    ---  ----------------------                          */
/*                                                                           */
/*    0.01    20/03/93  K.W  Creation                                        */
/*    1.00    05/08/96  K.W  Public release of DVE V1.0                      */
/*                                                                           */
/*****************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include <standard.h>

#include "tools.h"
#include "emu6809.h"
#include "dis6809.h"
#include "mon_cmds.h"
#include "keyboard.h"
#include "util.h"
#include "vdebug.h"
#include "verror.h"
#include "vectrex.h"
#include "sequence.h"
#include "emu6522.h"
#include "analog.h"
#include "sound.h"
#include "adlibsnd.h"
#include "sealsnd.h"
#include "display.h"
#include "monitor.h"


#ifdef fDebug
#undef fDebug
#endif
#define fDebug(level,body) ((bGblDebugEnable==T_TRUE && level>=tGblDebugLevel)?(tGblDebugMsgLevel=level),(fDebugXX body):0)

T_LISTE *bliste_first=NULL;
T_LISTE *bliste_last=NULL;
T_LISTE *wliste_first=NULL;
T_LISTE *wliste_last=NULL;

#define NO_RECAL_CHECK 0
#define INIT_RECAL_CHECK 1
#define INIT_RECAL_PAST1 2
#define INIT_RECAL_PAST2 3

T_INT recal_check=NO_RECAL_CHECK;
T_ULONG rc_cycles=0;

VARIABLE_LIST vlist[]=
{
 // 6809
 "6809_X",&tsCPURegisters.tX,T_WORD_LEN,
 "6809_Y",&tsCPURegisters.tY,T_WORD_LEN,
 "6809_D",&tsCPURegisters.uABD.tD,T_WORD_LEN,
 "6809_A",&tsCPURegisters.uABD.sAB.tA,T_BYTE_LEN,
 "6809_B",&tsCPURegisters.uABD.sAB.tB,T_BYTE_LEN,
 "6809_DP",&tsCPURegisters.tDP,T_WORD_LEN,
 "6809_PC",&tsCPURegisters.tPC,T_WORD_LEN,
 "6809_CC",&tsCPURegisters.uCC.tCC,T_WORD_LEN,
 "6809_USP",&tsCPURegisters.tUSP,T_WORD_LEN,
 "6809_HSP",&tsCPURegisters.tHSP,T_WORD_LEN,

 "6809_NMI",&tsCPURegisters.tNMIFlag,T_INT_LEN,
 "6809_IRQ",&tsCPURegisters.tIRQFlag,T_INT_LEN,
 "6809_FIRQ",&tsCPURegisters.tFIRQFlag,T_INT_LEN,

 // ANALOG STUFF
 "ANA_SIGDEF_SAMPHOLD",&tSIGDEF_SAMPHOLD,T_BYTE_LEN,
 "ANA_SIGDEF_AMUXSEL",&t6522ORB_1,T_BYTE_LEN,
 "ANA_SIGDEF_AMUXSEL",&t6522ORB_2,T_BYTE_LEN,
 "ANA_SIGDEF_COMPARE",&tSIGDEF_COMPARE,T_BYTE_LEN,
 "ANA_SIGDEF_RAMPB",&tSIGDEF_RAMPB,T_BYTE_LEN,
 "ANA_SIGDEF_XOUTPUT",&lSIGDEF_XOUTPUT,T_INT_LEN,
 "ANA_SIGDEF_YOUTPUT",&lSIGDEF_YOUTPUT,T_INT_LEN,
 "ANA_SIGDEF_ZOUTPUT",&tSIGDEF_ZOUTPUT,T_BYTE_LEN,
 "ANA_SIGDEF_BLANKB",&tSIGDEF_BLANKB,T_BYTE_LEN,
 "ANA_SIGDEF_ZEROB",&tSIGDEF_ZEROB,T_BYTE_LEN,
 "ANA_SIGDEF_DACBUS",&tSIGDEF_DACBUS,T_BYTE_LEN,
 "ANA_XSampleHold",&lXSampleHold,T_BYTE_LEN,
 "ANA_YSampleHold",&lYSampleHold,T_BYTE_LEN,
 "ANA_ZSampleHold",&tZSampleHold,T_BYTE_LEN,
 "ANA_LastRamp",&tLastRamp,T_BYTE_LEN,
 "ANA_IntegratorOffset",&tIntegratorOffset,T_BYTE_LEN,
 // 6522 STUFF
 "6522_ORB_0",&t6522ORB_0,T_BYTE_LEN,
 "6522_ORB_1",&t6522ORB_1,T_BYTE_LEN,
 "6522_ORB_2",&t6522ORB_2,T_BYTE_LEN,
 "6522_ORB_3",&t6522ORB_3,T_BYTE_LEN,
 "6522_ORB_4",&t6522ORB_4,T_BYTE_LEN,
 "6522_ORB_5",&t6522ORB_5,T_BYTE_LEN,
 "6522_ORB_6",&t6522ORB_6,T_BYTE_LEN,
 "6522_ORB_7",&t6522ORB_7,T_BYTE_LEN,
 "6522_IRB_0",&t6522IRB_0,T_BYTE_LEN,
 "6522_IRB_1",&t6522IRB_1,T_BYTE_LEN,
 "6522_IRB_2",&t6522IRB_2,T_BYTE_LEN,
 "6522_IRB_3",&t6522IRB_3,T_BYTE_LEN,
 "6522_IRB_4",&t6522IRB_4,T_BYTE_LEN,
 "6522_IRB_5",&t6522IRB_5,T_BYTE_LEN,
 "6522_IRB_6",&t6522IRB_6,T_BYTE_LEN,
 "6522_IRB_7",&t6522IRB_7,T_BYTE_LEN,
 "6522_IRB_0L",&t6522IRB_0L,T_BYTE_LEN,
 "6522_IRB_1L",&t6522IRB_1L,T_BYTE_LEN,
 "6522_IRB_2L",&t6522IRB_2L,T_BYTE_LEN,
 "6522_IRB_3L",&t6522IRB_3L,T_BYTE_LEN,
 "6522_IRB_4L",&t6522IRB_4L,T_BYTE_LEN,
 "6522_IRB_5L",&t6522IRB_5L,T_BYTE_LEN,
 "6522_IRB_6L",&t6522IRB_6L,T_BYTE_LEN,
 "6522_IRB_7L",&t6522IRB_7L,T_BYTE_LEN,
 "6522_ORA",&t6522ORA,T_BYTE_LEN,
 "6522_IRA",&t6522IRA,T_BYTE_LEN,
 "6522_IRAL",&t6522IRAL,T_BYTE_LEN,
 "6522_DBRA",&t6522DDRA,T_BYTE_LEN,
 "6522_DBRB",&t6522DDRB,T_BYTE_LEN,
 "6522_SHFT",&t6522SHFTREG,T_BYTE_LEN,
 "6522_AT1C",&t6522AUX_T1CTRL,T_BYTE_LEN,
 "6522_AT2C",&t6522AUX_T2CTRL,T_BYTE_LEN,
 "6522_ASHFTC",&t6522AUX_SHFTCTRL,T_BYTE_LEN,
 "6522_APALE",&t6522AUX_PA_LE,T_BYTE_LEN,
 "6522_APBLE",&t6522AUX_PB_LE,T_BYTE_LEN,
 "6522_PCRB1",&t6522PCR_CB1CTRL,T_BYTE_LEN,
 "6522_PCRB2",&t6522PCR_CB2CTRL,T_BYTE_LEN,
 "6522_PCRA1",&t6522PCR_CA1CTRL,T_BYTE_LEN,
 "6522_PCRA2",&t6522PCR_CA2CTRL,T_BYTE_LEN,
 "6522_IE",&t6522IER,T_BYTE_LEN,
 "6522_I",&t6522IFLG,T_BYTE_LEN,
 "6522_CA1",&t6522CA1,T_BYTE_LEN,
 "6522_CA2",&t6522CA2,T_BYTE_LEN,
 "6522_CB1",&t6522CB1,T_BYTE_LEN,
 "6522_CB2",&t6522CB2,T_BYTE_LEN,
 "6522_T1L",&u6522Timer1.uLatch.tW,T_WORD_LEN,
 "6522_T1C",&u6522Timer1.uCount.tW,T_WORD_LEN,
 "6522_T2L",&u6522Timer2.uLatch.tW,T_WORD_LEN,
 "6522_T2C",&u6522Timer2.uCount.tW,T_WORD_LEN,
 // PSG STUFF
 "SIGDEF_AY3BC1",&tSIGDEF_AY3BC1,T_BYTE_LEN,
 "SIGDEF_AY3BDIR",&tSIGDEF_AY3BDIR,T_BYTE_LEN,
 "PSG_0",&PSG_Reg[0],T_BYTE_LEN,
 "PSG_1",&PSG_Reg[1],T_BYTE_LEN,
 "PSG_2",&PSG_Reg[2],T_BYTE_LEN,
 "PSG_3",&PSG_Reg[3],T_BYTE_LEN,
 "PSG_4",&PSG_Reg[4],T_BYTE_LEN,
 "PSG_5",&PSG_Reg[5],T_BYTE_LEN,
 "PSG_6",&PSG_Reg[6],T_BYTE_LEN,
 "PSG_7",&PSG_Reg[7],T_BYTE_LEN,
 "PSG_8",&PSG_Reg[8],T_BYTE_LEN,
 "PSG_9",&PSG_Reg[9],T_BYTE_LEN,
 "PSG_a",&PSG_Reg[10],T_BYTE_LEN,
 "PSG_b",&PSG_Reg[11],T_BYTE_LEN,
 "PSG_c",&PSG_Reg[12],T_BYTE_LEN,
 "PSG_d",&PSG_Reg[13],T_BYTE_LEN,
 "PSG_0",&Regs[0],T_BYTE_LEN,
 "PSG_1",&Regs[1],T_BYTE_LEN,
 "PSG_2",&Regs[2],T_BYTE_LEN,
 "PSG_3",&Regs[3],T_BYTE_LEN,
 "PSG_4",&Regs[4],T_BYTE_LEN,
 "PSG_5",&Regs[5],T_BYTE_LEN,
 "PSG_6",&Regs[6],T_BYTE_LEN,
 "PSG_7",&Regs[7],T_BYTE_LEN,
 "PSG_8",&Regs[8],T_BYTE_LEN,
 "PSG_9",&Regs[9],T_BYTE_LEN,
 "PSG_a",&Regs[10],T_BYTE_LEN,
 "PSG_b",&Regs[11],T_BYTE_LEN,
 "PSG_c",&Regs[12],T_BYTE_LEN,
 "PSG_d",&Regs[13],T_BYTE_LEN,
 "PSG_imager",&imager_pulse_ticks,T_INT_LEN,
  NULL,0,0
};


T_UCHAR fDissi(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[]);
T_UCHAR fpause(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[]);
T_UCHAR frefresh(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[]);
T_UCHAR frecal_check(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[]);
T_UCHAR fcycles(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[]);
T_UCHAR fwatch(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[]);
T_UCHAR fpoke(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[]);
T_UCHAR flabels(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[]);

// #ifdef DEBUG_MON
#define MAX_COMMAND_LINE_ARGS 8
#define MAX_COMMAND_LINE_LENGTH EDIT_LINE_LEN
#define BACK_BUFFER_LENGTH 10

static T_ULONG last_bptick=0;

static T_INT x=-1;
static T_INT y=-1;
static T_INT u=-1;
static T_INT s=-1;
static T_INT d=-1;
static T_INT p=-1;
static T_INT a=-1;
static T_INT b=-1;
static T_INT c=-1;
static T_INT pg=-1;
static T_INT _nmi=-1;
static T_INT _irq=-1;
static T_INT _firq=-1;

static T_SLONG lOldXSH=-1;
static T_SLONG lOldYSH=-1;
static T_UCHAR tOldZSH=-1;

static T_SLONG lOldSigX=-1;
static T_SLONG lOldSigY=-1;
static T_UCHAR tOldSigZ=-1;

static T_UCHAR tOldBlank=0;
static T_UCHAR tOldRamp=123;
static T_UCHAR tOldZero=123;
static T_UCHAR tOldSel=123;
static T_UCHAR tOldShold=123;
static T_UCHAR tOldDac=123;
static T_UCHAR tOldComp=123;
static T_SCHAR tOldIntegratorOffset=-1;
static T_UCHAR tOldLastRamp=-1;

static T_UCHAR toldORB_0=0;
static T_UCHAR toldIRB_0=0;
static T_UCHAR toldIRB_0L=0;
static T_UCHAR toldORB_1=0;
static T_UCHAR toldIRB_1=0;
static T_UCHAR toldIRB_1L=0;
static T_UCHAR toldORB_2=0;
static T_UCHAR toldIRB_2=0;
static T_UCHAR toldIRB_2L=0;
static T_UCHAR toldORB_3=0;
static T_UCHAR toldIRB_3=0;
static T_UCHAR toldIRB_3L=0;
static T_UCHAR toldORB_4=0;
static T_UCHAR toldIRB_4=0;
static T_UCHAR toldIRB_4L=0;
static T_UCHAR toldORB_5=0;
static T_UCHAR toldIRB_5=0;
static T_UCHAR toldIRB_5L=0;
static T_UCHAR toldORB_6=0;
static T_UCHAR toldIRB_6=0;
static T_UCHAR toldIRB_6L=0;
static T_UCHAR toldORB_7=0;
static T_UCHAR toldIRB_7=0;
static T_UCHAR toldIRB_7L=0;
static T_UCHAR toldORA=0;
static T_UCHAR toldIRA=0;
static T_UCHAR toldIRAL=0;
static T_UCHAR toldDDRA=0;
static T_UCHAR toldDDRB=0;
static T_UCHAR toldSHFTREG=0;
static T_UCHAR toldAUX_T1CTRL=0;
static T_UCHAR toldAUX_T2CTRL=0;
static T_UCHAR toldAUX_SHFTCTRL=0;
static T_UCHAR toldAUX_PA_LE=0;
static T_UCHAR toldAUX_PB_LE=0;
static T_UCHAR toldPCR_CB1CTRL=0;
static T_UCHAR toldPCR_CB2CTRL=0;
static T_UCHAR toldPCR_CA1CTRL=0;
static T_UCHAR toldPCR_CA2CTRL=0;
static T_UCHAR toldIER=0;
static T_UCHAR toldIFLG=0;
static T_UCHAR toldCA1=0;
static T_UCHAR toldCA2=0;
static T_UCHAR toldCB1=0;
static T_UCHAR toldCB2=0;
static T_UWORD uoldTimer1uLatchtW=0;
static T_UWORD uoldTimer1uCounttW=0;
static T_UWORD uoldTimer2uLatchtW=0;
static T_UWORD uoldTimer2uCounttW=0;
static T_UCHAR _old0=0;
static T_UCHAR _old1=0;
static T_UCHAR _old2=0;
static T_UCHAR _old3=0;
static T_UCHAR _old4=0;
static T_UCHAR _old5=0;
static T_UCHAR _old6=0;
static T_UCHAR _old7=0;
static T_UCHAR _old8=0;
static T_UCHAR _old9=0;
static T_UCHAR _old10=0;
static T_UCHAR _old11=0;
static T_UCHAR _old12=0;
static T_UCHAR _old13=0;
static T_UCHAR _old14=0;
static T_UCHAR _old15=0;
#ifdef PATCH_OFFSET
static T_INT oldwait_ramp=T_FALSE;
#endif

static T_UWORD old_psg_reg[14]={1,1,1,1,1,1,1,1,1,1,1,1,1,1};
static T_UWORD tOldsamp=1;
static T_UWORD tOldamux=1;
static T_UWORD tOldaydir=1;
static T_UWORD tOldaybc=1;
static T_UWORD tOldOra_=1;
static T_UWORD tOldIra_=1;
static T_UWORD tOldKeyIn=1;
static T_UWORD tOldKeyOut=1;
static T_ULONG oldimager_pulse_ticks;

static T_CHAR string_buffer[EDIT_LINE_LEN+1];
static T_INT old_pause_state=T_FALSE;
static T_INT bStepRun=T_FALSE;
static T_INT bStep=T_FALSE;
static T_UBYTE last_acCommandLine[BACK_BUFFER_LENGTH][MAX_COMMAND_LINE_LENGTH+1];
static T_UINT buffer_in_pos=0;
static T_UINT buffer_out_pos=0;
T_INT emulator_update_frequency=10;


T_VOID watch_update_function(PARAMETER *parameter);
T_INT watch_destructor(PARAMETER *parameter);


T_INT monitor_destructor(PARAMETER *parameter);
T_INT monitor_interpreter(T_VOID *parameter);
T_INT monitor_history_back(T_VOID *parameter);
T_INT monitor_history_forward(T_VOID *parameter);
T_INT monitor_stop_srun(T_VOID *parameter);
T_INT monitor_exit(T_VOID *parameter);
T_VOID monitor_update_function(PARAMETER *parameter);
T_INT pause_on_monitor_entry=T_FALSE;
UPDATE_FUNCTION *monitor_update=NULL;
T_INT MonitorWinId=-1;

T_UCHAR fRegWin(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[]);
T_VOID register_update_function(PARAMETER *parameter);
T_INT register_destructor(PARAMETER *parameter);
T_INT open_register_window(T_VOID);
T_INT register_tasten(PARAMETER *parameter);
WINDOW_LISTEN_ELEMENT *RegWin=NULL;
T_INT RegWinId=-1;
T_INT regwininit=T_TRUE;

T_UCHAR fAnaWin(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[]);
T_VOID analog_update_function(PARAMETER *parameter);
T_INT analog_destructor(PARAMETER *parameter);
T_INT open_analog_window(T_VOID);
T_INT analog_tasten(PARAMETER *parameter);
T_VOID execute_analog_help(PARAMETER *parameter);
WINDOW_LISTEN_ELEMENT *AnaWin=NULL;
T_INT AnaWinId=-1;
T_INT anawininit=T_TRUE;

T_UCHAR f6522Win(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[]);
T_VOID _6522_update_function(PARAMETER *parameter);
T_INT _6522_destructor(PARAMETER *parameter);
T_INT open_6522_window(T_VOID);
T_INT _6522_tasten(PARAMETER *parameter);
T_VOID execute_6522_help(PARAMETER *parameter);
WINDOW_LISTEN_ELEMENT *_6522Win=NULL;
T_INT _6522WinId=-1;
T_INT _6522wininit=T_TRUE;

T_UCHAR fPSGWin(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[]);
T_VOID PSG_update_function(PARAMETER *parameter);
T_INT PSG_destructor(PARAMETER *parameter);
T_INT open_PSG_window(T_VOID);
T_INT PSG_tasten(PARAMETER *parameter);
T_VOID execute_PSG_help(PARAMETER *parameter);
WINDOW_LISTEN_ELEMENT *PSGWin=NULL;
T_INT PSGWinId=-1;
T_INT PSGwininit=T_TRUE;


// sizes MUST be correct for fixed font, otherwise garbage may appear!

WININFO asMonWin[RESOLUTION_COUNT]=
{
// Xpos Ypos Xsiz Ysiz
     0,   480-192-30,     592,   192, // 640  x 480     0
     0,   600-272-30,     672,   272, // 800  x 600     1
     0,   768-352-30,     752,   352, // 1024 x 768     2
     0,   1024-432-30,    832,   432, // 1280 x 1024    3
     0,   1200-512-30,    912,   512, // 1600 x 1200    4
     0,   200-80-30,      176,    80, // 320  x 200     5
};

/* All functions in the commandlist array accept and integer argument count */
/* and a pointer to and array of pointers to char which contain the args.   */
/* All functions return a BYTE flag T_TRUE/T_FALSE as an indication of if the   */
/* debug monitor should exit or not.                                        */

/* Define the command line array, all code for these commands can be found  */
/* in the file called mon_cmds.c                                            */

tsCOMMAND   asCommandList[]=
{
   NULL   ,"BSET"     ,"BSET xxxx,yyyy           - Set processor breakpoint for addr <xxxx> after <yyyy> (optional) passes",
   NULL   ,"BCLR"     ,"BCLR [i]                 - Clear all breakpoints (or optional variable breakpoint <i>)",
   NULL   ,"BSHOW"    ,"BSHOW                    - Show breakpoints",
   NULL   ,"DEBUG"    ,"DEBUG                    - Toggle debug window",
   NULL   ,"DBG"      ,"",
   NULL   ,"D"        ,"",
   NULL   ,"DUMP"     ,"D(UMP) xxxx,yyyy         - Memory Dump from <xxxx> to (<yyyy>) optional",
   NULL   ,"DR"       ,"DR                       - Dump 6809 registers",
   NULL   ,"F"        ,"",
   NULL   ,"FIND"     ,"F(IND) xxxx,yyyy <key>   - Find <key> hex or string between xxxx and yyyy",
   NULL   ,"H"        ,"",
   NULL   ,"HELP"     ,"H(ELP)                   - Print this text",
   NULL   ,"R"        ,"",
   NULL   ,"RUN"      ,"R(UN) xxxx               - Run the emulator from xxxx (optional)",
   NULL   ,"RESET"    ,"RESET <ALL|CPU|VIDEO>    - Reset vectrex hardware",
   NULL   ,"S"        ,"",
   NULL   ,"STEP"     ,"S(TEP) xx                - Single step the CPU <xx> instructions",
   NULL   ,"SRUN"     ,"SRUN xxxx                - Slow run & debug trace from <xxxx> (optional) (SPACE to stop)",
   NULL   ,"SEQ"      ,"SEQ s,yy                 - Exec <yy> steps of <s> (T=System,C=CPU,D=Display,S=Sound)",
   NULL   ,"T"        ,"",
   NULL   ,"TICKS"    ,"T(ICKS)                  - Display system tick counts",
   NULL   ,"Q"        ,"",
   NULL   ,"QUIT"     ,"QUIT                     - Quit from the Vectrex emulator back to DOS",
   NULL   ,"U"        ,"U xxxx,yy                - U xxxx yy  Un-assemble 6809 code from xxxx, yy instructions",
   NULL   ,"X"        ,"",
   NULL   ,"EXIT"     ,"EXIT|X                   - Exit the debug monitor",
   NULL   ,"REGWIN"   ,"REGWIN                   - Open Register information Window",
   NULL   ,"ANAWIN"   ,"ANAWIN                   - Open Analog information Window",
   NULL   ,"6522WIN"  ,"6522WIN                  - Open 6522 information Window",
   NULL   ,"PSGWIN"   ,"PSGWIN                   - Open AY-3-8192 information Window",
   NULL   ,"SOURCE"   ,"SOURCE                   - Output Source code",
   NULL   ,"B_SET"    ,"B_SET string|addr ,value - Set breakpoint for reg<string>|mem<addr> on value <value>",
   NULL   ,"PAUSE"    ,"PAUSE                    - Toggles PAUSE state",
   NULL   ,"REFRESH"  ,"REFRESH                  - refresh windows",
   NULL   ,"RC"       ,"",
   NULL   ,"RECAL_CHECK" ,"RECAL_CHECK              - Get number of cycles between exit and enter of wait_recal",
   NULL   ,"C"  ,"",
   NULL   ,"CYCLES"   ,"CYCLES [set_value]       - Set or get cycles since last set",
   NULL   ,"W"  ,"",
   NULL   ,"WATCH"    ,"WATCH string|addr        - Create watch window for reg<string>|mem<addr>",
   NULL   ,"P"  ,"",
   NULL   ,"POKE"     ,"POKE string|addr, value  - Poke <value> to reg<string>|mem<addr>",
   NULL   ,"L"        ,"",
   NULL   ,"LABELS"     ,"LABELS, Lists all valid <strings>",
   NULL   ,""         ,""
};

/*****************************************************************************/
/*                                                                           */
/*   Function    : fDebugMonitor()                                           */
/*                                                                           */
/*   Author      : K.W                                                       */
/*                                                                           */
/*   Description : This function is the runtime debugger and is called from  */
/*                 the sequencer if the debugger is enabled once on every    */
/*                 loop of the sequencer. It has a number of local flags     */
/*                 that are used for breakpointing the emulator. The func    */
/*                 then either returns or gets user input from a command     */
/*                 line interpreter. See the command list for details of     */
/*                 what commands are available.                              */
/*                 All i/o is done via the functions contained in io.c       */
/*                                                                           */
/*                                                                           */
/*   Accepts     : nothing                                                   */
/*                                                                           */
/*   Returns     : nothing                                                   */
/*                                                                           */
/*****************************************************************************/

T_INT open_monitor_window(T_VOID)
{
 static T_INT in_mon=0;
 WINDOW_LISTEN_ELEMENT *MonitorWin;
 OPEN_WIN w;
 T_INT i;
 if (get_window(MonitorWinId))
  return T_TRUE;
 w.x_pos=asMonWin[wGblDisplayMode].iWinXpos;
 w.y_pos=asMonWin[wGblDisplayMode].iWinYpos;
 w.breite=asMonWin[wGblDisplayMode].iWinXsize;
 w.hoehe=asMonWin[wGblDisplayMode].iWinYsize;
 w.border=2;
 w.color=ddark_grau;
 w.bcolor=dhell_grau;
 w.art=RAW_WINDOW|WITH_CLOSE_BUTTON|WITH_HIDE_BUTTON|WITH_DRAG_BUTTON|VIRTUAL_FRONT_WINDOW|REMEMBER|EDIT_TRUE;
 w.id=WIN_OPEN_CHOOSES_ID;
 w.variablen=NULL;
 w.mem_bild=NULL;
 w.pcx_bild=NULL;
 w.name="Monitor Window for DVE";
 w.destructor=monitor_destructor;
 w.area_funktion=execute_activate;
 w.tasten_funktion=NULL;
 MOUSE_VERSTECKE
 MonitorWin=open_window(&w);
 if (MonitorWin==NULL)
 {
  show_message("Monitor Window opening error!");
  MOUSE_ZEIGE
  return T_FALSE;
 }
 MonitorWinId=MonitorWin->id;
 print_srect(MonitorWin,PRINT_WIN_RAW,0,0,MonitorWin->b,MonitorWin->h,ddark_grau,weiss,2);
 edit(MonitorWin,"monitor.tmp",EDIT_EDIT|EDIT_START|EDIT_LOG|EDIT_NO_LOG_FILE);
 insert_edit_key_funktion( MonitorWin, KEY_ENTER, AFTER_EDIT, monitor_interpreter);
 insert_edit_key_funktion( MonitorWin, KEY_CTRL_UP, BEFORE_EDIT, monitor_history_back);
 insert_edit_key_funktion( MonitorWin, KEY_CTRL_DOWN, BEFORE_EDIT, monitor_history_forward);
 insert_edit_key_funktion( MonitorWin, KEY_SPACE, BEFORE_EDIT, monitor_stop_srun);
 insert_edit_key_funktion( MonitorWin, KEY_ALT_M, BEFORE_EDIT, monitor_exit);
 edit_set_help_table(MonitorWin,"monitor.key");

 monitor_update=insert_update_funktion(monitor_update_function,1);
 if (!in_mon)
 {
  for (i=0;i<BACK_BUFFER_LENGTH;i++)
   strcpy(last_acCommandLine[i],"QUIT");
  buffer_in_pos=0;
  buffer_out_pos=0;
  in_mon=1;
 }
 fDebug(HIGH,("Monitor enabled."));
 fDebug(HIGH,("Single CPU processing enabled."));
 edit_log_add(MonitorWin,"Vectrex HP3000 Emulator Debug Monitor");
 edit_log_add(MonitorWin,"-------------------------------------");
 edit_log_add(MonitorWin,"");
 edit_log_add(MonitorWin,"Type h(elp) for the command list");
 edit_log_add(MonitorWin,"");
 bGblForceSingleProcessor=T_TRUE;
 MOUSE_ZEIGE
 asCommandList[0].pfCommandFunc=fMonCmdBreakSet;
 asCommandList[1].pfCommandFunc=fMonCmdBreakClear;
 asCommandList[2].pfCommandFunc=fMonCmdBreakShow;
 asCommandList[3].pfCommandFunc=fMonCmdDebug;
 asCommandList[4].pfCommandFunc=fMonCmdDebug;
 asCommandList[5].pfCommandFunc=fMonCmdDump;
 asCommandList[6].pfCommandFunc=fMonCmdDump;
 asCommandList[7].pfCommandFunc=fMonCmdRegDump;
 asCommandList[8].pfCommandFunc=fMonCmdFind;
 asCommandList[9].pfCommandFunc=fMonCmdFind;
 asCommandList[10].pfCommandFunc=fMonCmdHelp;
 asCommandList[11].pfCommandFunc=fMonCmdHelp;
 asCommandList[12].pfCommandFunc=fMonCmdRun;
 asCommandList[13].pfCommandFunc=fMonCmdRun;
 asCommandList[14].pfCommandFunc=fMonCmdReset;
 asCommandList[15].pfCommandFunc=fMonCmdStep;
 asCommandList[16].pfCommandFunc=fMonCmdStep;
 asCommandList[17].pfCommandFunc=fMonCmdStepRun;
 asCommandList[18].pfCommandFunc=fMonCmdSequence;
 asCommandList[19].pfCommandFunc=fMonCmdTicks;
 asCommandList[20].pfCommandFunc=fMonCmdTicks;
 asCommandList[21].pfCommandFunc=fMonCmdQuit;
 asCommandList[22].pfCommandFunc=fMonCmdQuit;
 asCommandList[23].pfCommandFunc=fMonCmdDissemble;
 asCommandList[24].pfCommandFunc=fMonCmdExit;
 asCommandList[25].pfCommandFunc=fMonCmdExit;
 asCommandList[26].pfCommandFunc=fRegWin;
 asCommandList[27].pfCommandFunc=fAnaWin;
 asCommandList[28].pfCommandFunc=f6522Win;
 asCommandList[29].pfCommandFunc=fPSGWin;
 asCommandList[30].pfCommandFunc=fDissi;
 asCommandList[31].pfCommandFunc=fBset;
 asCommandList[32].pfCommandFunc=fpause;
 asCommandList[33].pfCommandFunc=frefresh;
 asCommandList[34].pfCommandFunc=frecal_check;
 asCommandList[35].pfCommandFunc=frecal_check;
 asCommandList[36].pfCommandFunc=fcycles;
 asCommandList[37].pfCommandFunc=fcycles;
 asCommandList[38].pfCommandFunc=fwatch;
 asCommandList[39].pfCommandFunc=fwatch;
 asCommandList[40].pfCommandFunc=fpoke;
 asCommandList[41].pfCommandFunc=fpoke;
 asCommandList[42].pfCommandFunc=flabels;
 asCommandList[43].pfCommandFunc=flabels;
 old_pause_state=T_FALSE;
 if (bGblPauseMode==T_TRUE)
 {
  old_pause_state=T_TRUE;
  pause_on_monitor_entry=T_TRUE;
 }
 bGblPauseMode=T_FALSE;
 vectrex_update->frequency=1;
 return T_TRUE;
}

T_VOID close_monitor_window(T_VOID)
{
 WINDOW_LISTEN_ELEMENT *MonitorWin=get_window(MonitorWinId);
 if (MonitorWin!=NULL)
 {
  MonitorWin->parameter=NULL; // might not be zero, but must forced, since it
                              // is a none returning destructor using window
  close_window(MonitorWin,CLOSE_DEFAULT);
 }
}

T_INT monitor_destructor(PARAMETER *parameter)
{
 WINDOW_LISTEN_ELEMENT *window=(WINDOW_LISTEN_ELEMENT *)parameter->window_element;
 if (parameter==NULL)
 {
  show_message("No Parameter in DESTRUCTOR()");
  return -1;
 }
 if (window==NULL)
 {
  show_message("I can't find myself!!! (Window not closed)");
  return -1;
 }
// interesting part of destructor (just a notifier in this case)
 bGblDebugMonitorEnable=T_FALSE;
 remove_update_funktion(monitor_update_function);
 monitor_update=NULL;
 fDebug(HIGH,("Single CPU processing disabled."));
 fDebug(HIGH,("Monitor disabled."));
 bGblPauseMode=old_pause_state;
 bGblForceSingleProcessor=T_FALSE;
 vectrex_update->frequency=emulator_update_frequency;
 pause_on_monitor_entry=pause_on_monitor_entry_default;
 window->destructor=NULL; // the rest is done by close_window
 window->parameter=NULL;
 MonitorWinId=-1;
 return -1; // keine variablen da, dann ebene = -1
}

T_INT monitor_history_back(T_VOID *parameter)
{
 EDIT_ELEMENT *ed=(EDIT_ELEMENT *)parameter;
 EDIT_ZEILE *zeile;
 if (buffer_out_pos==buffer_in_pos)
 {
  /* first back buffer, add current line to buffer */
  strcpy(last_acCommandLine[buffer_out_pos],ed->cursor_line->text);
 }
 if (!buffer_out_pos)
  buffer_out_pos+=BACK_BUFFER_LENGTH;
 buffer_out_pos--;
 if (buffer_out_pos==buffer_in_pos)
 {
  /* buffer overflow... */
  buffer_out_pos=(buffer_out_pos+1)%BACK_BUFFER_LENGTH;
  return EDIT_DO_NOTHING;
 }
 /* now replace last line in ed with the buffer position */
 zeile=(EDIT_ZEILE *)(ed->text_zeile_letzte->objekt);
 strcpy(zeile->text,last_acCommandLine[buffer_out_pos]);
 zeile->len_chars=strlen(zeile->text);
 zeile->len_pixel=(zeile->len_chars)*ed->len_x;
 return EDIT_CURSOR_TO_END|EDIT_REDRAW_LINE|EDIT_DO_NOTHING;
}

T_INT monitor_history_forward(T_VOID *parameter)
{
 EDIT_ELEMENT *ed=(EDIT_ELEMENT *)parameter;
 EDIT_ZEILE *zeile;
 if (buffer_out_pos==buffer_in_pos)
 {
  /* nothing in buffer yet */
  return EDIT_DO_NOTHING;
 }
 buffer_out_pos=(buffer_out_pos+1)%BACK_BUFFER_LENGTH;
 zeile=(EDIT_ZEILE *)(ed->text_zeile_letzte->objekt);
 strcpy(zeile->text,last_acCommandLine[buffer_out_pos]);
 zeile->len_chars=strlen(zeile->text);
 zeile->len_pixel=(zeile->len_chars)*ed->len_x;
 return EDIT_CURSOR_TO_END|EDIT_REDRAW_LINE|EDIT_DO_NOTHING;
}

T_VOID monitor_update_function(PARAMETER *parameter)
{
 T_UBYTE  acCommandLine[MAX_COMMAND_LINE_LENGTH+1];
 T_PUBYTE apcArgV[MAX_COMMAND_LINE_ARGS];
 T_UWORD  iBPoint,iBPSkip,iBPEnable;
 T_LISTE *bpl=bliste_first;
 CPUREGISTERS tsTmp;
 T_INT status6809=f6809StatusRead();
 if ((status6809!=CPU_FETCH) && (status6809!=CPU_WAIT) && (status6809!=CPU_CWAI) && (status6809!=CPU_SYNC))
 {
  fDebug(LOW,("fDebugMonitor() rejected monitor entrance 0x%02x",status6809));
  return;
 }

 // Decide depending on what has happened - cases are:
 //
 //  0.  Recal Check
 //  1.  Monitor open
 //  2.  SystemTickBreakpoint
 //  3.  Instruction stepping
 //  4.  CPU exception
 //  5.  Steprun
 //  6.  Breakpoint
 //  7.  Variable Breakpoint
 //

//  0.  Recal Check
 if (recal_check!=NO_RECAL_CHECK)
 {
  if (recal_check==INIT_RECAL_CHECK)
  {
   if (f6809LastInsAddr()==0xF192)
   {
    recal_check=INIT_RECAL_PAST1;
    return;
   }
  }
  else if (recal_check==INIT_RECAL_PAST1)
  {
   if (f6809LastInsAddr()==0xF1A2)
   {
    rc_cycles=lwGblSystemTickTotal;
    recal_check=INIT_RECAL_PAST2;
    return;
   }
  }
  else if (recal_check==INIT_RECAL_PAST2)
  {
   if (f6809LastInsAddr()==0xF192)
   {
    rc_cycles=lwGblSystemTickTotal-rc_cycles;
    recal_check=NO_RECAL_CHECK;
    sprintf(string_buffer,"Wait_Recal ticks from exit($f1a0) to enter($f192):%ld",rc_cycles);
    edit_log_add(get_window(MonitorWinId),string_buffer);
    edit_log_add(get_window(MonitorWinId),"");
    vector_undraw();
    vector_refresh();
    pause_on_monitor_entry=T_FALSE;
    bStepRun=T_FALSE;
    bStep=T_FALSE;
    bGblPauseMode=T_TRUE;
    frefresh(0,NULL,NULL);
    return;
   }
  }
 }

 //  1.  Monitor open
 if (pause_on_monitor_entry==T_TRUE)
 {
  fMonCmdTicks(0,NULL,NULL);
  fMonCmdRegDump(0,NULL,NULL);
  pause_on_monitor_entry=T_FALSE;
  bStepRun=T_FALSE;
  bStep=T_FALSE;
  bGblPauseMode=T_TRUE;
  vector_undraw();
  vector_refresh();
  return;
 }

 //  2.  SystemTickBreakpoint
 /* 1st Check all the breakpoints to see if we should run command line */
 if (lwGblSystemTickBreakpoint!=0)
 {
  if (lwGblSystemTickTotal>=lwGblSystemTickBreakpoint)
  {
   if (MonitorWinId!=-1)
    if (get_window(MonitorWinId)->hide)
     show_raw_window(get_window(MonitorWinId));
   fDebug(HIGH,("fDebugMonitor(): SystemTick breakpoint detected at %ld",lwGblSystemTickBreakpoint));
   fMonCmdTicks(0,NULL,NULL);
   sprintf(acCommandLine,"0x%04x",f6809LastInsAddr());
   apcArgV[0]=acCommandLine;
   apcArgV[1]=acCommandLine;
   apcArgV[2]="0";
   fMonCmdDissemble(3,apcArgV,apcArgV);
   fMonCmdRegDump(0,NULL,NULL);
   sprintf(acCommandLine,"0x%04x",tsTmp.tPC);
   apcArgV[0]=acCommandLine;
   apcArgV[1]=acCommandLine;
   apcArgV[2]="0";
   fMonCmdDissemble(3,apcArgV,apcArgV);
   lwGblSystemTickBreakpoint=0;
   bGblPauseMode=T_TRUE;
   sprintf(string_buffer,"Ticks since last break:%08ld",lwGblSystemTickTotal-last_bptick);
   edit_log_add(get_window(MonitorWinId),string_buffer);
   edit_log_add(get_window(MonitorWinId),"");
   last_bptick=lwGblSystemTickTotal;
   vector_undraw();
   vector_refresh();
   frefresh(0,NULL,NULL);
   return;
  }
  else
  {
   return;
  }
 }

 f6809RegistersRead(&tsTmp);
 //  3.  Instruction stepping
 if ((bStep==T_TRUE)&&(f6809LastInsAddr()!=tsTmp.tPC))
 {
  steps--;
  if (steps==0)
  {
   if (MonitorWinId!=-1)
    if (get_window(MonitorWinId)->hide)
     show_raw_window(get_window(MonitorWinId));
   fDebug(HIGH,("fDebugMonitor(): Stepping done"));
   fMonCmdTicks(0,NULL,NULL);
   sprintf(acCommandLine,"0x%04x",f6809LastInsAddr());
   apcArgV[0]=acCommandLine;
   apcArgV[1]=acCommandLine;
   apcArgV[2]="0";
   fMonCmdDissemble(3,apcArgV,apcArgV);
   fMonCmdRegDump(0,NULL,NULL);
   bStep=T_FALSE;
   bGblPauseMode=T_TRUE;
   vector_undraw();
   vector_refresh();
   return;
  }
 }

 //  4.  CPU exception
 if ((status6809==CPU_WAIT)&&(gblException==T_TRUE))
 {
  gblException=T_FALSE;
  //*** fWinPrintf(pwMonitorWin,asMonWin[wGblDisplayMode].iWinFontsize,"\n**** 6809 Exception generated at address $%04x ****\n",f6809LastInsAddr());
  if (MonitorWinId!=-1)
   if (get_window(MonitorWinId)->hide)
    show_raw_window(get_window(MonitorWinId));
  sprintf(string_buffer,"**** 6809 Exception generated at address $%04x ****",f6809LastInsAddr());
  edit_log_add(get_window(MonitorWinId),string_buffer);
  edit_log_add(get_window(MonitorWinId),"");
  fMonCmdTicks(0,NULL,NULL);
  sprintf(acCommandLine,"0x%04x",f6809LastInsAddr());
  apcArgV[0]=acCommandLine;
  apcArgV[1]=acCommandLine;
  apcArgV[2]="0";
  fMonCmdDissemble(3,apcArgV,apcArgV);
  fMonCmdRegDump(0,NULL,NULL);
  lwGblSystemTickBreakpoint=0;
  bStep=T_FALSE;
  bStepRun=T_FALSE;
  bGblPauseMode=T_TRUE;
  vector_undraw();
  vector_refresh();
  frefresh(0,NULL,NULL);
  return;
 }

 //  5.  Steprun
 if ((f6809LastInsAddr()!=tsTmp.tPC) && (bStepRun==T_TRUE))
 {
  if (MonitorWinId!=-1)
   if (get_window(MonitorWinId)->hide)
    show_raw_window(get_window(MonitorWinId));
  fMonCmdTicks(0,NULL,NULL);
  sprintf(acCommandLine,"0x%04x",f6809LastInsAddr());
  apcArgV[0]=acCommandLine;
  apcArgV[1]=acCommandLine;
  apcArgV[2]="0";
  fMonCmdDissemble(3,apcArgV,apcArgV);
  fMonCmdRegDump(0,NULL,NULL);
  return;
 }

 //  6.  Breakpoint
 f6809ReadBreakpoint(&iBPoint,&iBPSkip,&iBPEnable);
 if ((iBPoint==tsTmp.tPC) && (iBPSkip==0) && (iBPEnable==T_TRUE) && (bGblPauseMode==T_FALSE))
 {
  if (MonitorWinId!=-1)
   if (get_window(MonitorWinId)->hide)
    show_raw_window(get_window(MonitorWinId));
  f6809ClearBreakpoint();
  sprintf(string_buffer,"**** 6809 Breakpoint occured at address $%04x ****",tsTmp.tPC);
  edit_log_add(get_window(MonitorWinId),string_buffer);
  edit_log_add(get_window(MonitorWinId),"");
  fMonCmdTicks(0,NULL,NULL);
  fMonCmdRegDump(0,NULL,NULL);
  sprintf(acCommandLine,"0x%04x",tsTmp.tPC);
  apcArgV[0]=acCommandLine;
  apcArgV[1]=acCommandLine;
  apcArgV[2]="0";
  fMonCmdDissemble(3,apcArgV,apcArgV);
  bGblPauseMode=T_TRUE;
  sprintf(string_buffer,"Ticks since last break:%08ld",lwGblSystemTickTotal-last_bptick);
  edit_log_add(get_window(MonitorWinId),string_buffer);
  edit_log_add(get_window(MonitorWinId),"");
  last_bptick=lwGblSystemTickTotal;
  vector_undraw();
  vector_refresh();
  bStepRun=T_FALSE;
  frefresh(0,NULL,NULL);
  return;
 }
 //  7. Variable Breakpoint
 while (bpl)
 {
  VARIABLE_BREAKPOINT *bp=(VARIABLE_BREAKPOINT *)bpl->objekt;
  if (bp)
  {
   T_UINT value;
   if (bp->len==T_BYTE_LEN)
    value=(T_UINT)(*((T_PUCHAR)bp->pointer));
   if (bp->len==T_WORD_LEN)
    value=(T_UINT)(*((T_PUWORD)bp->pointer));
   if (bp->len==T_INT_LEN)
    value=(T_UINT)(*((T_PUINT)bp->pointer));
   if (bp->kind&T_KIND_BREAK_ON_VALUE)
   {
    if (value==bp->compare)
    {
     if (bp->kind&T_KIND_REG)
      sprintf(string_buffer,"**** 6809 Breakpoint occured, %s == %i",bp->name,bp->compare);
     if (bp->kind&T_KIND_MEM)
     {
      if (bp->kind&T_KIND_LABEL)
      {
       sprintf(string_buffer,"**** 6809 Breakpoint occured, %s (%i($%04X)) == %i",bp->name,(T_UWORD)(((T_PUCHAR)bp->pointer)-(&atSystemImage[0])),(T_UWORD)(((T_PUCHAR)bp->pointer)-(&atSystemImage[0])),bp->compare);
      }
      else
      {
       sprintf(string_buffer,"**** 6809 Breakpoint occured, %i($%04X) == %i",(T_UWORD)(((T_PUCHAR)bp->pointer)-(&atSystemImage[0])),(T_UWORD)(((T_PUCHAR)bp->pointer)-(&atSystemImage[0])),bp->compare);
      }
     }
     d_entferne_aus_liste(bp,&bliste_first,&bliste_last,DL_CLEAN_UP);
     if (bp)
      free(bp);
     if (MonitorWinId!=-1)
      if (get_window(MonitorWinId)->hide)
       show_raw_window(get_window(MonitorWinId));
     edit_log_add(get_window(MonitorWinId),string_buffer);
     edit_log_add(get_window(MonitorWinId),"");
     fMonCmdTicks(0,NULL,NULL);
     fMonCmdRegDump(0,NULL,NULL);
     sprintf(acCommandLine,"0x%04x",tsTmp.tPC);
     apcArgV[0]=acCommandLine;
     apcArgV[1]=acCommandLine;
     apcArgV[2]="0";
     fMonCmdDissemble(3,apcArgV,apcArgV);
     bGblPauseMode=T_TRUE;
     sprintf(string_buffer,"Ticks since last break:%08ld",lwGblSystemTickTotal-last_bptick);
     edit_log_add(get_window(MonitorWinId),string_buffer);
     edit_log_add(get_window(MonitorWinId),"");
     last_bptick=lwGblSystemTickTotal;
     vector_undraw();
     vector_refresh();
     bStepRun=T_FALSE;
     frefresh(0,NULL,NULL);
     return;
    }
   }
   else if (bp->kind&T_KIND_BREAK_ON_CHANGE)
   {
    if (value!=bp->start)
    {
     if (bp->kind&T_KIND_REG)
      sprintf(string_buffer,"**** 6809 Breakpoint occured, %s was %i, now %i",bp->name,bp->start,value);
     if (bp->kind&T_KIND_MEM)
     {
      if (bp->kind&T_KIND_LABEL)
      {
       sprintf(string_buffer,"**** 6809 Breakpoint occured, %s (%i($%04X)) was %i, now %i",bp->name,(T_UWORD)(((T_PUCHAR)bp->pointer)-(&atSystemImage[0])),(T_UWORD)(((T_PUCHAR)bp->pointer)-(&atSystemImage[0])),bp->start,value);
      }
      else
      {
       sprintf(string_buffer,"**** 6809 Breakpoint occured, %i($%04X) was %i, now %i",(T_UWORD)(((T_PUCHAR)bp->pointer)-(&atSystemImage[0])),(T_UWORD)(((T_PUCHAR)bp->pointer)-(&atSystemImage[0])),bp->start,value);
      }
     }
     if (MonitorWinId!=-1)
      if (get_window(MonitorWinId)->hide)
       show_raw_window(get_window(MonitorWinId));
     edit_log_add(get_window(MonitorWinId),string_buffer);
     edit_log_add(get_window(MonitorWinId),"");
     fMonCmdTicks(0,NULL,NULL);
     fMonCmdRegDump(0,NULL,NULL);
     sprintf(acCommandLine,"0x%04x",tsTmp.tPC);
     apcArgV[0]=acCommandLine;
     apcArgV[1]=acCommandLine;
     apcArgV[2]="0";
     fMonCmdDissemble(3,apcArgV,apcArgV);
     bGblPauseMode=T_TRUE;
     sprintf(string_buffer,"Ticks since last break:%08ld",lwGblSystemTickTotal-last_bptick);
     edit_log_add(get_window(MonitorWinId),string_buffer);
     edit_log_add(get_window(MonitorWinId),"");
     last_bptick=lwGblSystemTickTotal;
     vector_undraw();
     vector_refresh();
     bStepRun=T_FALSE;
     bp->start=value;
     frefresh(0,NULL,NULL);
     return;
    }
   }
  }
  bpl=bpl->naechstes;
 }

 return;
}

T_INT monitor_stop_srun(T_VOID *parameter)
{
 if (bStepRun==T_TRUE)
 {
  bStepRun=T_FALSE;
  bGblPauseMode=T_TRUE;
  vector_undraw();
  vector_refresh();
  return EDIT_DO_NOTHING;
 }
 if (bStep==T_TRUE)
 {
  steps=1;
  return EDIT_DO_NOTHING;
 }
 return EDIT_CONTINUE;
}

T_INT monitor_exit(T_VOID *parameter)
{
 close_monitor_window();
 return EDIT_QUIT;
}

/***************************************************************/
/***** Register Information Window *****************************/
/***************************************************************/
/* Only default.font supported 8*16 */
/************************************/

T_INT register_destructor(PARAMETER *parameter)
{
 WINDOW_LISTEN_ELEMENT *window=(WINDOW_LISTEN_ELEMENT *)parameter->window_element;
 if (parameter==NULL)
 {
  show_message("No Parameter in register_destructor()");
  return -1;
 }
 if (window==NULL)
 {
  show_message("I can't find myself!!! (Register Window not closed)");
  return -1;
 }
// interesting part of destructor (just a notifier in this case)
 remove_update_funktion(register_update_function);
 window->destructor=NULL; // the rest is done by close_window
 RegWinId=-1;
 RegWin=NULL;
 return -1; // keine variablen da, dann ebene = -1
}

T_VOID register_update_function(PARAMETER *parameter)
{
 static T_LONG old_pos=-1;
 CPUREGISTERS tsTmp;
 T_INT old_print_modus;
 T_INT old_back_color;
 T_INT old_text_color;
 T_INT old_text;

 if (old_pos==lwGblSystemTickTotal)
  return;
 old_pos=lwGblSystemTickTotal;
 if (RegWin->hide!=0)
  return;

 f6809RegistersRead(&tsTmp);
 old_print_modus=vesa_print_modus;
 old_text=vesa_text;
 old_back_color=vesa_back_color;
 old_text_color=vesa_text_color;
 vesa_print_modus=PIC_FILLED;
 vesa_text=SINGLE_COLOR;
 vesa_back_color=dhell_grau;
 vesa_text_color=ddark_grau;

 t_hide_win_mouse(RegWin);
 if (((x!=tsTmp.tX)||(y!=tsTmp.tY))||(parameter==1))
  print_with_all_features(RegWin,PRINT_CLIPPED_OC,2+4*8,   2,PMAX,"%04X Y :$%04X",tsTmp.tX,tsTmp.tY);
 if (((s!=tsTmp.tUSP)||(u!=tsTmp.tHSP))||(parameter==1))
  print_with_all_features(RegWin,PRINT_CLIPPED_OC,2+4*8,  20,PMAX,"%04X H :$%04X",tsTmp.tUSP,tsTmp.tHSP);
 if (((a!=tsTmp.uABD.sAB.tA)||(b!=tsTmp.uABD.sAB.tB)||(p!=tsTmp.tPC))||(parameter==1))
  print_with_all_features(RegWin,PRINT_CLIPPED_OC,2+4*8,  38,PMAX,"%04X PC:$%04X",tsTmp.uABD.tD,tsTmp.tPC);
 if (((a!=tsTmp.uABD.sAB.tA)||(pg!=tsTmp.tDP))||(parameter==1))
  print_with_all_features(RegWin,PRINT_CLIPPED_OC,2+4*8,  56,PMAX,"%02X   DP:$%04X",tsTmp.uABD.sAB.tA,tsTmp.tDP);
 if (((b!=tsTmp.uABD.sAB.tB)||(c!=tsTmp.uCC.tCC))||(parameter==1))
  print_with_all_features(RegWin,PRINT_CLIPPED_OC,2+4*8,  74,PMAX,"%02X   CC:$%02X",tsTmp.uABD.sAB.tB,tsTmp.uCC.tCC);
 if ((c!=tsTmp.uCC.tCC)||(parameter==1))
  print_with_all_features(RegWin,PRINT_CLIPPED_OC,2,      92,PMAX,"%d%d%d%d%d%d%d%d",tsTmp.uCC.sCC.bC,tsTmp.uCC.sCC.bV,tsTmp.uCC.sCC.bZ,tsTmp.uCC.sCC.bN,tsTmp.uCC.sCC.bI,tsTmp.uCC.sCC.bH,tsTmp.uCC.sCC.bF,
                          tsTmp.uCC.sCC.bF,tsTmp.uCC.sCC.bE);
 if (wGblDisplayMode<5)
 {
  if ((_nmi!=tsCPURegisters.tNMIFlag)||(parameter==1))
   print_with_all_features(RegWin,PRINT_CLIPPED_OC,2+5*8,128,PMAX,"%i",tsCPURegisters.tNMIFlag);
  if ((_irq!=tsCPURegisters.tIRQFlag)||(parameter==1))
   print_with_all_features(RegWin,PRINT_CLIPPED_OC,2+5*8,144,PMAX,"%i",tsCPURegisters.tIRQFlag);
  if ((_firq!=tsCPURegisters.tFIRQFlag)||(parameter==1))
   print_with_all_features(RegWin,PRINT_CLIPPED_OC,2+5*8,160,PMAX,"%i",tsCPURegisters.tFIRQFlag);
 }
 t_show_win_mouse(RegWin);

 vesa_print_modus=old_print_modus;
 vesa_text=old_text;
 vesa_back_color=old_back_color;
 vesa_text_color=old_text_color;

 x=tsTmp.tX;
 y=tsTmp.tY;
 u=tsTmp.tUSP;
 s=tsTmp.tHSP;
 d=tsTmp.uABD.tD;
 p=tsTmp.tPC;
 a=tsTmp.uABD.sAB.tA;
 b=tsTmp.uABD.sAB.tB;
 c=tsTmp.uCC.tCC;
 pg=tsTmp.tDP;

 _nmi=tsCPURegisters.tNMIFlag;
 _irq=tsCPURegisters.tIRQFlag;
 _firq=tsCPURegisters.tFIRQFlag;

 if (regwininit==T_TRUE)
 {
  if (get_window(MonitorWinId))
   aktivate_window(get_window(MonitorWinId));
  regwininit=T_FALSE;
 }
 return;
}

T_INT open_register_window(T_VOID)
{
 OPEN_WIN w;
 CPUREGISTERS tsTmp;
 T_INT old_print_modus;
 T_INT old_text;
 T_INT old_back_color;
 T_INT old_text_color;
 if (get_window(RegWinId))
  return T_TRUE;
 w.x_pos=0;
 w.y_pos=0;
 w.breite=17*8+4;
 w.hoehe=11*16;
 w.border=2;
 w.color=ddark_grau;
 w.bcolor=dhell_grau;
 w.art=RAW_WINDOW|WITH_CLOSE_BUTTON|WITH_HIDE_BUTTON|WITH_DRAG_BUTTON|REMEMBER;//|VIRTUAL_FRONT_WINDOW;
 w.id=WIN_OPEN_CHOOSES_ID;
 w.variablen=NULL;
 w.mem_bild=NULL;
 w.pcx_bild=NULL;
 w.name="6809Regs";
 w.destructor=register_destructor;
 w.area_funktion=execute_activate;
 w.tasten_funktion=register_tasten;
 MOUSE_VERSTECKE
 RegWin=open_window(&w);
 if (RegWin==NULL)
 {
  show_message("Register Window opening error!");
  MOUSE_ZEIGE
  return T_FALSE;
 }
 RegWinId=RegWin->id;
 print_srect(RegWin,PRINT_WIN_RAW,0,0,RegWin->b,RegWin->h,ddark_grau,weiss,2);

 if (get_window(MonitorWinId))
 {
  remove_update_funktion(monitor_update_function);
  insert_update_funktion(register_update_function,1);
  monitor_update=insert_update_funktion(monitor_update_function,1);
 }
 else
  insert_update_funktion(register_update_function,1);

 old_print_modus=vesa_print_modus;
 old_text=vesa_text;
 old_back_color=vesa_back_color;
 old_text_color=vesa_text_color;
 vesa_print_modus=PIC_FILLED;
 vesa_text=SINGLE_COLOR;
 vesa_back_color=dhell_grau;
 vesa_text_color=ddark_grau;

 wprint(RegWin,2,   2,"X :$");
 wprint(RegWin,2,  20,"U :$");
 wprint(RegWin,2,  38,"D :$");
 wprint(RegWin,2,  56,"A :$");
 wprint(RegWin,2,  74,"B :$");
 wprint(RegWin,2, 110,"CVZNIHFE");
 if (wGblDisplayMode<5)
 {
  wprint(RegWin,2, 128,"NMI :");
  wprint(RegWin,2, 144,"IRQ :");
  wprint(RegWin,2, 160,"FIRQ:");
 }
 f6809RegistersRead(&tsTmp);


 print_with_all_features(RegWin,PRINT_RAW_OC,2+4*8+RegWin->x+RegWin->x_0,   2+RegWin->y+RegWin->y_0,PMAX,"%04X Y :$%04X",tsTmp.tX,tsTmp.tY);
 print_with_all_features(RegWin,PRINT_RAW_OC,2+4*8+RegWin->x+RegWin->x_0,  20+RegWin->y+RegWin->y_0,PMAX,"%04X H :$%04X",tsTmp.tUSP,tsTmp.tHSP);
 print_with_all_features(RegWin,PRINT_RAW_OC,2+4*8+RegWin->x+RegWin->x_0,  38+RegWin->y+RegWin->y_0,PMAX,"%04X PC:$%04X",tsTmp.uABD.tD,tsTmp.tPC);
 print_with_all_features(RegWin,PRINT_RAW_OC,2+4*8+RegWin->x+RegWin->x_0,  56+RegWin->y+RegWin->y_0,PMAX,"%02X   DP:$%04X",tsTmp.uABD.sAB.tA,tsTmp.tDP);
 print_with_all_features(RegWin,PRINT_RAW_OC,2+4*8+RegWin->x+RegWin->x_0,  74+RegWin->y+RegWin->y_0,PMAX,"%02X   CC:$%02X",tsTmp.uABD.sAB.tB,tsTmp.uCC.tCC);
 print_with_all_features(RegWin,PRINT_RAW_OC,2    +RegWin->x+RegWin->x_0,  92+RegWin->y+RegWin->y_0,PMAX,"%d%d%d%d%d%d%d%d",
                          tsTmp.uCC.sCC.bC,tsTmp.uCC.sCC.bV,
                          tsTmp.uCC.sCC.bZ,tsTmp.uCC.sCC.bN,
                          tsTmp.uCC.sCC.bI,tsTmp.uCC.sCC.bH,
                          tsTmp.uCC.sCC.bF,tsTmp.uCC.sCC.bF,
                          tsTmp.uCC.sCC.bE);
 if (wGblDisplayMode<5)
 {
  print_with_all_features(RegWin,PRINT_RAW_OC,2+5*8+RegWin->x+RegWin->x_0,128+RegWin->y+RegWin->y_0,PMAX,"%i",tsCPURegisters.tNMIFlag);
  print_with_all_features(RegWin,PRINT_RAW_OC,2+5*8+RegWin->x+RegWin->x_0,144+RegWin->y+RegWin->y_0,PMAX,"%i",tsCPURegisters.tIRQFlag);
  print_with_all_features(RegWin,PRINT_RAW_OC,2+5*8+RegWin->x+RegWin->x_0,160+RegWin->y+RegWin->y_0,PMAX,"%i",tsCPURegisters.tFIRQFlag);
 }

 update_whole_window(RegWin);
 vesa_print_modus=old_print_modus;
 vesa_text=old_text;
 vesa_back_color=old_back_color;
 vesa_text_color=old_text_color;
 MOUSE_ZEIGE
 return T_TRUE;
}

T_UCHAR fRegWin(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[])
{
 regwininit=T_TRUE;
 open_register_window();
 return 0;
}

T_INT register_tasten(PARAMETER *parameter)
{
 if (get_window(MonitorWinId))
  aktivate_window(get_window(MonitorWinId));
 if (bStep==T_TRUE)
 {
  steps=1;
 }
 return 0;
}

/***************************************************************/
/***** Analog Information Window *******************************/
/***************************************************************/
/* Only default.font supported 8*16 */
/************************************/

T_INT analog_tasten(PARAMETER *parameter)
{
 if (get_window(MonitorWinId))
  aktivate_window(get_window(MonitorWinId));
 if (bStep==T_TRUE)
 {
  steps=1;
 }
 return 0;
}

T_INT analog_destructor(PARAMETER *parameter)
{
 WINDOW_LISTEN_ELEMENT *window=(WINDOW_LISTEN_ELEMENT *)parameter->window_element;
 if (parameter==NULL)
 {
  show_message("No Parameter in analog_destructor()");
  return -1;
 }
 if (window==NULL)
 {
  show_message("I can't find myself!!! (Analog Window not closed)");
  return -1;
 }
// interesting part of destructor (just a notifier in this case)
 remove_update_funktion(analog_update_function);
 window->destructor=NULL; // the rest is done by close_window
 AnaWinId=-1;
 AnaWin=NULL;
 return -1; // keine variablen da, dann ebene = -1
}

T_VOID analog_update_function(PARAMETER *parameter)
{
 static T_LONG old_pos=-1;
 T_INT old_print_modus;
 T_INT old_back_color;
 T_INT old_text_color;
 T_INT old_text;
 if (old_pos==lwGblSystemTickTotal)
  return;
 old_pos=lwGblSystemTickTotal;
 if (AnaWin->hide!=0)
  return;

 old_print_modus=vesa_print_modus;
 old_text=vesa_text;
 old_back_color=vesa_back_color;
 old_text_color=vesa_text_color;
 vesa_print_modus=PIC_FILLED;
 vesa_text=SINGLE_COLOR;
 vesa_back_color=dhell_grau;
 vesa_text_color=ddark_grau;

 t_hide_win_mouse(AnaWin);
 if ((tOldShold!=tSIGDEF_SAMPHOLD)||(parameter==1))
  print_with_all_features(AnaWin,PRINT_CLIPPED_OC,2+(18+4)*8,   2+ 0*18,PMAX,"%03i",tSIGDEF_SAMPHOLD);
 if ((tOldSel!=tSIGDEF_AMUXSEL)||(parameter==1))
  print_with_all_features(AnaWin,PRINT_CLIPPED_OC,2+(18+4)*8,   2+ 1*18,PMAX,"%03i",tSIGDEF_AMUXSEL);
 if ((tOldComp!=tSIGDEF_COMPARE)||(parameter==1))
  print_with_all_features(AnaWin,PRINT_CLIPPED_OC,2+(18+4)*8 ,  2+ 2*18,PMAX,"%03i",tSIGDEF_COMPARE);
 if ((tOldRamp!=tSIGDEF_RAMPB)||(parameter==1))
  print_with_all_features(AnaWin,PRINT_CLIPPED_OC,2+(18+4)*8,   2+ 3*18,PMAX,"%03i",tSIGDEF_RAMPB);
 if ((lOldSigX!=lSIGDEF_XOUTPUT)||(parameter==1))
  print_with_all_features(AnaWin,PRINT_CLIPPED_OC,2+18*8,       2+ 4*18,PMAX,"%07i",lSIGDEF_XOUTPUT);
 if ((lOldSigY!=lSIGDEF_YOUTPUT)||(parameter==1))
  print_with_all_features(AnaWin,PRINT_CLIPPED_OC,2+18*8,       2+ 5*18,PMAX,"%07i",lSIGDEF_YOUTPUT);
 if ((tOldSigZ!=tSIGDEF_ZOUTPUT)||(parameter==1))
  print_with_all_features(AnaWin,PRINT_CLIPPED_OC,2+(18+4)*8,   2+ 6*18,PMAX,"%03i",tSIGDEF_ZOUTPUT);
 if ((tOldBlank!=tSIGDEF_BLANKB)||(parameter==1))
  print_with_all_features(AnaWin,PRINT_CLIPPED_OC,2+(18+4)*8,   2+ 7*18,PMAX,"%03i",tSIGDEF_BLANKB);
 if ((tOldZero!=tSIGDEF_ZEROB)||(parameter==1))
  print_with_all_features(AnaWin,PRINT_CLIPPED_OC,2+(18+4)*8,   2+ 8*18,PMAX,"%03i",tSIGDEF_ZEROB);
 if (wGblDisplayMode<5)
 {
  if ((tOldDac!=tSIGDEF_DACBUS)||(parameter==1))
   print_with_all_features(AnaWin,PRINT_CLIPPED_OC,2+(18+3)*8,  2+ 9*18,PMAX,"%04i",(T_SCHAR)tSIGDEF_DACBUS);
  if ((lOldXSH!=lXSampleHold)||(parameter==1))
   print_with_all_features(AnaWin,PRINT_CLIPPED_OC,2+18*8,      2+10*18,PMAX,"%07i",lXSampleHold);
  if ((lOldYSH!=lYSampleHold)||(parameter==1))
   print_with_all_features(AnaWin,PRINT_CLIPPED_OC,2+18*8,      2+11*18,PMAX,"%07i",lYSampleHold);
  if ((tOldZSH!=tZSampleHold)||(parameter==1))
   print_with_all_features(AnaWin,PRINT_CLIPPED_OC,2+(18+4)*8,  2+12*18,PMAX,"%03i",tZSampleHold);
  if ((tOldLastRamp!=tLastRamp)||(parameter==1))
   print_with_all_features(AnaWin,PRINT_CLIPPED_OC,2+18*8,      2+13*18,PMAX,"%07i",tLastRamp);
  if ((tOldIntegratorOffset!=tIntegratorOffset)||(parameter==1))
   print_with_all_features(AnaWin,PRINT_CLIPPED_OC,2+18*8,      2+14*18,PMAX,"%07i",tIntegratorOffset);
 }
 t_show_win_mouse(AnaWin);

 vesa_print_modus=old_print_modus;
 vesa_text=old_text;
 vesa_back_color=old_back_color;
 vesa_text_color=old_text_color;

 tOldIntegratorOffset=tIntegratorOffset;
 tOldLastRamp=tLastRamp;

 lOldXSH=lXSampleHold;
 lOldYSH=lYSampleHold;
 tOldZSH=tZSampleHold;

 lOldSigX=lSIGDEF_XOUTPUT;
 lOldSigY=lSIGDEF_YOUTPUT;
 tOldSigZ=tSIGDEF_ZOUTPUT;

 tOldBlank=tSIGDEF_BLANKB;
 tOldRamp=tSIGDEF_RAMPB;
 tOldZero=tSIGDEF_ZEROB;
 tOldSel=tSIGDEF_AMUXSEL;
 tOldShold=tSIGDEF_SAMPHOLD;
 tOldDac=tSIGDEF_DACBUS;
 tOldComp=tSIGDEF_COMPARE;

 if (anawininit==T_TRUE)
 {
  if (get_window(MonitorWinId))
   aktivate_window(get_window(MonitorWinId));
  anawininit=T_FALSE;
 }
 return;
}

T_INT open_analog_window(T_VOID)
{
 OPEN_WIN w;
 T_INT old_print_modus;
 T_INT old_text;
 T_INT old_back_color;
 T_INT old_text_color;
 if (get_window(AnaWinId))
  return T_TRUE;
 w.x_pos=0;
 w.y_pos=0;
 w.breite=25*8+4;
 w.hoehe=15*18;
 w.border=2;
 w.color=ddark_grau;
 w.bcolor=dhell_grau;
 w.art=RAW_WINDOW|WITH_CLOSE_BUTTON|WITH_HIDE_BUTTON|WITH_DRAG_BUTTON|REMEMBER;//|VIRTUAL_FRONT_WINDOW;
 w.id=WIN_OPEN_CHOOSES_ID;
 w.variablen=NULL;
 w.mem_bild=NULL;
 w.pcx_bild=NULL;
 w.name="Analog";
 w.destructor=analog_destructor;
 w.area_funktion=execute_analog_help;
 w.tasten_funktion=analog_tasten;
 MOUSE_VERSTECKE
 AnaWin=open_window(&w);
 if (AnaWin==NULL)
 {
  show_message("Analog Window opening error!");
  MOUSE_ZEIGE
  return T_FALSE;
 }
 AnaWinId=AnaWin->id;
 print_srect(AnaWin,PRINT_WIN_RAW,0,0,AnaWin->b,AnaWin->h,ddark_grau,weiss,2);

 if (get_window(MonitorWinId))
 {
  remove_update_funktion(monitor_update_function);
  insert_update_funktion(analog_update_function,1);
  monitor_update=insert_update_funktion(monitor_update_function,1);
 }
 else
  insert_update_funktion(analog_update_function,1);
 old_print_modus=vesa_print_modus;
 old_text=vesa_text;
 old_back_color=vesa_back_color;
 old_text_color=vesa_text_color;
 vesa_print_modus=PIC_FILLED;
 vesa_text=SINGLE_COLOR;
 vesa_back_color=dhell_grau;
 vesa_text_color=ddark_grau;

 wprint(AnaWin,2,  2+0*18, "SIGDEF_SAMPHOLD :");
 wprint(AnaWin,2,  2+1*18, "SIGDEF_AMUXSEL  :");
 wprint(AnaWin,2,  2+2*18, "SIGDEF_COMPARE  :");
 wprint(AnaWin,2,  2+3*18, "SIGDEF_RAMPB    :");
 wprint(AnaWin,2,  2+4*18, "SIGDEF_XOUTPUT  :");
 wprint(AnaWin,2,  2+5*18, "SIGDEF_YOUTPUT  :");
 wprint(AnaWin,2,  2+6*18, "SIGDEF_ZOUTPUT  :");
 wprint(AnaWin,2,  2+7*18, "SIGDEF_BLANKB   :");
 wprint(AnaWin,2,  2+8*18, "SIGDEF_ZEROB    :");
 if (wGblDisplayMode<5)
 {
  wprint(AnaWin,2,  2+ 9*18,"SIGDEF_DACBUS   :");
  wprint(AnaWin,2,  2+10*18,"XSampleHold     :");
  wprint(AnaWin,2,  2+11*18,"YSampleHold     :");
  wprint(AnaWin,2,  2+12*18,"ZSampleHold     :");
  wprint(AnaWin,2,  2+13*18,"LastRamp        :");
  wprint(AnaWin,2,  2+14*18,"IntegratorOffset:");
 }
 print_with_all_features(AnaWin,PRINT_RAW_OC,2+(18+4)*8+AnaWin->x+AnaWin->x_0,   2+ 0*18+AnaWin->y+AnaWin->y_0,PMAX,"%03i",tSIGDEF_SAMPHOLD);
 print_with_all_features(AnaWin,PRINT_RAW_OC,2+(18+4)*8+AnaWin->x+AnaWin->x_0,   2+ 1*18+AnaWin->y+AnaWin->y_0,PMAX,"%03i",tSIGDEF_AMUXSEL);
 print_with_all_features(AnaWin,PRINT_RAW_OC,2+(18+4)*8+AnaWin->x+AnaWin->x_0,  2+  2*18+AnaWin->y+AnaWin->y_0,PMAX,"%03i",tSIGDEF_COMPARE);
 print_with_all_features(AnaWin,PRINT_RAW_OC,2+(18+4)*8+AnaWin->x+AnaWin->x_0,   2+ 3*18+AnaWin->y+AnaWin->y_0,PMAX,"%03i",tSIGDEF_RAMPB);
 print_with_all_features(AnaWin,PRINT_RAW_OC,2+18*8+AnaWin->x+AnaWin->x_0,       2+ 4*18+AnaWin->y+AnaWin->y_0,PMAX,"%07i",lSIGDEF_XOUTPUT);
 print_with_all_features(AnaWin,PRINT_RAW_OC,2+18*8+AnaWin->x+AnaWin->x_0,       2+ 5*18+AnaWin->y+AnaWin->y_0,PMAX,"%07i",lSIGDEF_YOUTPUT);
 print_with_all_features(AnaWin,PRINT_RAW_OC,2+(18+4)*8+AnaWin->x+AnaWin->x_0,   2+ 6*18+AnaWin->y+AnaWin->y_0,PMAX,"%03i",tSIGDEF_ZOUTPUT);
 print_with_all_features(AnaWin,PRINT_RAW_OC,2+(18+4)*8+AnaWin->x+AnaWin->x_0,   2+ 7*18+AnaWin->y+AnaWin->y_0,PMAX,"%03i",tSIGDEF_BLANKB);
 print_with_all_features(AnaWin,PRINT_RAW_OC,2+(18+4)*8+AnaWin->x+AnaWin->x_0,   2+ 8*18+AnaWin->y+AnaWin->y_0,PMAX,"%03i",tSIGDEF_ZEROB);
 if (wGblDisplayMode<5)
 {
  print_with_all_features(AnaWin,PRINT_RAW_OC,2+(18+3)*8+AnaWin->x+AnaWin->x_0,   2+ 9*18+AnaWin->y+AnaWin->y_0,PMAX,"%04i",(T_SCHAR)tSIGDEF_DACBUS);
  print_with_all_features(AnaWin,PRINT_RAW_OC,2+18*8+AnaWin->x+AnaWin->x_0,       2+10*18+AnaWin->y+AnaWin->y_0,PMAX,"%07i",lXSampleHold);
  print_with_all_features(AnaWin,PRINT_RAW_OC,2+18*8+AnaWin->x+AnaWin->x_0,       2+11*18+AnaWin->y+AnaWin->y_0,PMAX,"%07i",lYSampleHold);
  print_with_all_features(AnaWin,PRINT_RAW_OC,2+(18+4)*8+AnaWin->x+AnaWin->x_0,   2+12*18+AnaWin->y+AnaWin->y_0,PMAX,"%03i",tZSampleHold);
  print_with_all_features(AnaWin,PRINT_RAW_OC,2+18*8+AnaWin->x+AnaWin->x_0,       2+13*18+AnaWin->y+AnaWin->y_0,PMAX,"%07i",tLastRamp);
  print_with_all_features(AnaWin,PRINT_RAW_OC,2+18*8+AnaWin->x+AnaWin->x_0,       2+14*18+AnaWin->y+AnaWin->y_0,PMAX,"%07i",tIntegratorOffset);
 }
 update_whole_window(AnaWin);
 vesa_print_modus=old_print_modus;
 vesa_text=old_text;
 vesa_back_color=old_back_color;
 vesa_text_color=old_text_color;
 MOUSE_ZEIGE
 return T_TRUE;
}

T_UCHAR fAnaWin(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[])
{
 anawininit=T_TRUE;
 open_analog_window();
 return 0;
}

T_VOID execute_analog_help(PARAMETER *parameter)
{
 WINDOW_LISTEN_ELEMENT *window=(WINDOW_LISTEN_ELEMENT *)parameter->window_element;
 if (parameter==NULL)
 {
  return;
 }
 if (window==NULL)
 {
  return;
 }
 if (parameter->maus_button==1)
 {
  parameter->executed=PARAMETER_EXECUTED;
  aktivate_window(window);
 }
 if (parameter->maus_button==2)
 {
  /* set cursor to this position */
//  T_INT maus_y=parameter->maus_y-(window->y+window->y_0);
//  maus_y+=(font_now->zeichen[' '].hoehe-1);
//  maus_y/=(font_now->zeichen[' '].hoehe);
  T_INT maus_y=parameter->maus_y-(window->y+window->y_0);
  parameter->executed=PARAMETER_EXECUTED;
  maus_y+=17;
  maus_y/=18;
  switch (maus_y)
  {
   case 1:
   {
    help_function("analog\\sigshold.hlf",NULL);
    break;
   }
   case 2:
   {
    help_function("analog\\sigasel.hlf",NULL);
    break;
   }
   case 3:
   {
    help_function("analog\\sigcom.hlf",NULL);
    break;
   }
   case 4:
   {
    help_function("analog\\sigramp.hlf",NULL);
    break;
   }
   case 5:
   {
    help_function("analog\\xout.hlf",NULL);
    break;
   }
   case 6:
   {
    help_function("analog\\yout.hlf",NULL);
    break;
   }
   case 7:
   {
    help_function("analog\\zout.hlf",NULL);
    break;
   }
   case 8:
   {
    help_function("analog\\sigblank.hlf",NULL);
    break;
   }
   case 9:
   {
    help_function("analog\\sigzero.hlf",NULL);
    break;
   }
   case 10:
   {
    help_function("analog\\sigdac.hlf",NULL);
    break;
   }
   case 11:
   {
    help_function("analog\\xsample.hlf",NULL);
    break;
   }
   case 12:
   {
    help_function("analog\\ysample.hlf",NULL);
    break;
   }
   case 13:
   {
    help_function("analog\\zsample.hlf",NULL);
    break;
   }
   case 14:
   {
    help_function("analog\\lastramp.hlf",NULL);
    break;
   }
   case 15:
   {
    help_function("analog\\ioff.hlf",NULL);
    break;
   }
  }
 }
}

/***************************************************************/
/***** 6522 Information Window *********************************/
/***************************************************************/
/* Only default.font supported 8*16 */
/************************************/

T_INT _6522_tasten(PARAMETER *parameter)
{
 if (get_window(MonitorWinId))
  aktivate_window(get_window(MonitorWinId));
 if (bStep==T_TRUE)
 {
  steps=1;
 }
 return 0;
}
T_INT _6522_destructor(PARAMETER *parameter)
{
 WINDOW_LISTEN_ELEMENT *window=(WINDOW_LISTEN_ELEMENT *)parameter->window_element;
 if (parameter==NULL)
 {
  show_message("No Parameter in _6522_destructor()");
  return -1;
 }
 if (window==NULL)
 {
  show_message("I can't find myself!!! (6522 Window not closed)");
  return -1;
 }
// interesting part of destructor (just a notifier in this case)
 remove_update_funktion(_6522_update_function);
 window->destructor=NULL; // the rest is done by close_window
 _6522WinId=-1;
 _6522Win=NULL;
 return -1; // keine variablen da, dann ebene = -1
}

T_VOID _6522_update_function(PARAMETER *parameter)
{
 static T_LONG old_pos=-1;
 volatile static T_UCHAR _0;
 volatile static T_UCHAR _1;
 volatile static T_UCHAR _2;
 volatile static T_UCHAR _3;
 volatile static T_UCHAR _4;
 volatile static T_UCHAR _5;
 volatile static T_UCHAR _6;
 volatile static T_UCHAR _7;
 volatile static T_UCHAR _8;
 volatile static T_UCHAR _9;
 volatile static T_UCHAR _10;
 volatile static T_UCHAR _11;
 volatile static T_UCHAR _12;
 volatile static T_UCHAR _13;
 volatile static T_UCHAR _14;
 volatile static T_UCHAR _15;
 T_INT old_print_modus;
 T_INT old_back_color;
 T_INT old_text_color;
 T_INT old_text;

 if (old_pos==lwGblSystemTickTotal)
  return;
 old_pos=lwGblSystemTickTotal;
 if (_6522Win->hide!=0)
  return;
 _0=
 ((t6522ORB_0 & TMPL_RB_0)|
  (t6522ORB_1 & TMPL_RB_1)|
  (t6522ORB_2 & TMPL_RB_2)|
  (t6522ORB_3 & TMPL_RB_3)|
  (t6522ORB_4 & TMPL_RB_4)|
  (t6522ORB_5 & TMPL_RB_5)|
  (t6522ORB_6 & TMPL_RB_6)|
  (t6522ORB_7 & TMPL_RB_7));

 _1=(t6522ORA&t6522DDRA)|((t6522AUX_PA_LE)?(t6522IRAL&(t6522DDRA^0xff)):(t6522IRA&(t6522DDRA^0xff)));
 _2=(t6522DDRB);
 _3=(t6522DDRA);
 _4=(u6522Timer1.uCount.sW.tL);
 _5=(u6522Timer1.uCount.sW.tH);
 _6=(u6522Timer1.uLatch.sW.tL);
 _7=(u6522Timer1.uLatch.sW.tH);
 _8=(u6522Timer2.uCount.sW.tL);
 _9=(u6522Timer2.uCount.sW.tH);
 _10=(t6522SHFTREG);
 _11=(t6522AUX_T1CTRL<<6) | (t6522AUX_T2CTRL<<5) | (t6522AUX_SHFTCTRL<<2) | (t6522AUX_PB_LE<<1) | t6522AUX_PA_LE;;
 _12=(t6522PCR_CB2CTRL<<5) | (t6522PCR_CB1CTRL<<4) | (t6522PCR_CA2CTRL<<1) | t6522PCR_CA1CTRL;
 _13=((t6522IFLG&t6522IER)?0x80:0)|(t6522IFLG);
 _14=(0x80|t6522IER);
 _15=(t6522ORA&t6522DDRA)|(t6522IRA&(t6522DDRA^0xff));;

 old_print_modus=vesa_print_modus;
 old_text=vesa_text;
 old_back_color=vesa_back_color;
 old_text_color=vesa_text_color;
 vesa_print_modus=PIC_FILLED;
 vesa_text=SINGLE_COLOR;
 vesa_back_color=dhell_grau;
 vesa_text_color=ddark_grau;

 t_hide_win_mouse(_6522Win);
 if (((t6522ORB_0!=toldORB_0)||(t6522IRB_0!=toldIRB_0)||(t6522IRB_0L!=toldIRB_0L)||(_0!=_old0))||(parameter==1))
  print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,       2+ 0*18 ,PMAX,"$%02X IRB_0:$%02X IRB_0L:$%02X 0:$%02X",t6522ORB_0,t6522IRB_0,t6522IRB_0L,_0);
 if (((t6522ORB_1!=toldORB_1)||(t6522IRB_1!=toldIRB_1)||(t6522IRB_1L!=toldIRB_1L)||(_1!=_old1))||(parameter==1))
  print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,       2+ 1*18 ,PMAX,"$%02X IRB_1:$%02X IRB_1L:$%02X 1:$%02X",t6522ORB_1,t6522IRB_1,t6522IRB_1L,_1);
 if (((t6522ORB_2!=toldORB_2)||(t6522IRB_2!=toldIRB_2)||(t6522IRB_2L!=toldIRB_2L)||(_2!=_old2))||(parameter==1))
  print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,       2+ 2*18 ,PMAX,"$%02X IRB_2:$%02X IRB_2L:$%02X 2:$%02X",t6522ORB_2,t6522IRB_2,t6522IRB_2L,_2);
 if (((t6522ORB_3!=toldORB_3)||(t6522IRB_3!=toldIRB_3)||(t6522IRB_3L!=toldIRB_3L)||(_3!=_old3))||(parameter==1))
  print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,       2+ 3*18 ,PMAX,"$%02X IRB_3:$%02X IRB_3L:$%02X 3:$%02X",t6522ORB_3,t6522IRB_3,t6522IRB_3L,_3);
 if (((t6522ORB_4!=toldORB_4)||(t6522IRB_4!=toldIRB_4)||(t6522IRB_4L!=toldIRB_4L)||(_4!=_old4))||(parameter==1))
  print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,       2+ 4*18 ,PMAX,"$%02X IRB_4:$%02X IRB_4L:$%02X 4:$%02X",t6522ORB_4,t6522IRB_4,t6522IRB_4L,_4);
 if (((t6522ORB_5!=toldORB_5)||(t6522IRB_5!=toldIRB_5)||(t6522IRB_5L!=toldIRB_5L)||(_5!=_old5))||(parameter==1))
  print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,       2+ 5*18 ,PMAX,"$%02X IRB_5:$%02X IRB_5L:$%02X 5:$%02X",t6522ORB_5,t6522IRB_5,t6522IRB_5L,_5);
 if (((t6522ORB_6!=toldORB_6)||(t6522IRB_6!=toldIRB_6)||(t6522IRB_6L!=toldIRB_6L)||(_6!=_old6))||(parameter==1))
  print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,       2+ 6*18 ,PMAX,"$%02X IRB_6:$%02X IRB_6L:$%02X 6:$%02X",t6522ORB_6,t6522IRB_6,t6522IRB_6L,_6);
 if (((t6522ORB_7!=toldORB_7)||(t6522IRB_7!=toldIRB_7)||(t6522IRB_7L!=toldIRB_7L)||(_7!=_old7))||(parameter==1))
  print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,       2+ 7*18 ,PMAX,"$%02X IRB_7:$%02X IRB_7L:$%02X 7:$%02X",t6522ORB_7,t6522IRB_7,t6522IRB_7L,_7);
 if (((t6522ORA!=toldORA)||(t6522IRA!=toldIRA)||(t6522IRAL!=toldIRAL)||(_8!=_old8))||(parameter==1))
  print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,       2+ 8*18 ,PMAX,"$%02X IRA  :$%02X IRAL  :$%02X 8:$%02X",t6522ORA,t6522IRA,t6522IRAL,_8);
 if (wGblDisplayMode<5)
 {
  if (((t6522DDRA!=toldDDRA)||(t6522DDRB!=toldDDRB)||(t6522SHFTREG!=toldSHFTREG)||(_9!=_old9))||(parameter==1))
   print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,   2+ 9*18 ,PMAX,"$%02X DDRB :$%02X SHFT  :$%02X 9:$%02X",t6522DDRA,t6522DDRB,t6522SHFTREG,_9);
  if (((t6522AUX_T1CTRL!=toldAUX_T2CTRL)||(t6522AUX_T2CTRL!=toldAUX_T2CTRL)||(t6522AUX_SHFTCTRL!=toldAUX_SHFTCTRL)||(_10!=_old10))||(parameter==1))
   print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,   2+10*18 ,PMAX,"$%02X AT2C :$%02X ASHFTC:$%02X a:$%02X",t6522AUX_T1CTRL,t6522AUX_T2CTRL,t6522AUX_SHFTCTRL,_10);
  if (((t6522AUX_PA_LE!=toldAUX_PA_LE)||(t6522AUX_PB_LE!=toldAUX_PB_LE)||(_11!=_old11))||(parameter==1))
   print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,   2+11*18 ,PMAX,"$%02X APBLE:$%02X            b:$%02X",t6522AUX_PA_LE,t6522AUX_PB_LE,_11);
  if (((t6522PCR_CB1CTRL!=toldPCR_CB1CTRL)||(t6522PCR_CB2CTRL!=toldPCR_CB2CTRL)||(_12!=_old12))||(parameter==1))
   print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,   2+12*18 ,PMAX,"$%02X PCRB2:$%02X            c:$%02X",t6522PCR_CB1CTRL,t6522PCR_CB2CTRL,_12);
  if (((t6522PCR_CA1CTRL!=toldPCR_CA1CTRL)||(t6522PCR_CA2CTRL!=toldPCR_CA2CTRL)||(_13!=_old13))||(parameter==1))
   print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,   2+13*18 ,PMAX,"$%02X PCRA2:$%02X            d:$%02X",t6522PCR_CA1CTRL,t6522PCR_CA2CTRL,_13);
  if (((t6522IER!=toldIER)||(t6522IFLG!=toldIFLG)||(_14!=_old14))||(parameter==1))
   print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,   2+14*18 ,PMAX,"$%02X I    :$%02X            e:$%02X",t6522IER,t6522IFLG,_14);
  if (((t6522CA1!=toldCA1)||(t6522CA2!=toldCA1)||(_15!=_old15))||(parameter==1))
   print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,   2+15*18 ,PMAX,"$%02X CA2  :$%02X            f:$%02X",t6522CA1,t6522CA2,_15);
  if (((t6522CB1!=toldCB1)||(t6522CB2!=toldCB1))||(parameter==1))
   print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,   2+16*18 ,PMAX,"$%02X CB2  :$%02X",t6522CB1,t6522CB2);
  if (((u6522Timer1.uLatch.tW!=uoldTimer1uLatchtW)||(u6522Timer1.uCount.tW!=uoldTimer1uCounttW))||(parameter==1))
   print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,   2+17*18 ,PMAX,"$%04X T1C:$%04X",u6522Timer1.uLatch.tW,u6522Timer1.uCount.tW);
  if (((u6522Timer2.uLatch.tW!=uoldTimer2uLatchtW)||(u6522Timer2.uCount.tW!=uoldTimer2uCounttW))||(parameter==1))
   print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+6*8 ,   2+18*18 ,PMAX,"$%04X T2C:$%04X",u6522Timer2.uLatch.tW,u6522Timer2.uCount.tW);
#ifdef PATCH_OFFSET
  vesa_text_color=f_red;
  if (oldwait_ramp!=wait_ramp)
  {
   if (wait_ramp==T_WAIT_WAS_PATCHED)
    print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+24*8 , 2+18*18 ,PMAX,"PATCHING");
   else
    print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+24*8 , 2+18*18 ,PMAX,"        ");
  }
#endif
 }
 t_show_win_mouse(_6522Win);

 vesa_print_modus=old_print_modus;
 vesa_text=old_text;
 vesa_back_color=old_back_color;
 vesa_text_color=old_text_color;
 _old0=_0;
 _old1=_1;
 _old2=_2;
 _old3=_3;
 _old4=_4;
 _old5=_5;
 _old6=_6;
 _old7=_7;
 _old8=_8;
 _old9=_9;
 _old10=_10;
 _old11=_11;
 _old12=_12;
 _old13=_13;
 _old14=_14;
 _old15=_15;
 toldORB_0=t6522ORB_0;
 toldIRB_0=t6522IRB_0;
 toldIRB_0L=t6522IRB_0L;
 toldORB_1=t6522ORB_1;
 toldIRB_1=t6522IRB_1;
 toldIRB_1L=t6522IRB_1L;
 toldORB_2=t6522ORB_2;
 toldIRB_2=t6522IRB_2;
 toldIRB_2L=t6522IRB_2L;
 toldORB_3=t6522ORB_3;
 toldIRB_3=t6522IRB_3;
 toldIRB_3L=t6522IRB_3L;
 toldORB_4=t6522ORB_4;
 toldIRB_4=t6522IRB_4;
 toldIRB_4L=t6522IRB_4L;
 toldORB_5=t6522ORB_5;
 toldIRB_5=t6522IRB_5;
 toldIRB_5L=t6522IRB_5L;
 toldORB_6=t6522ORB_6;
 toldIRB_6=t6522IRB_6;
 toldIRB_6L=t6522IRB_6L;
 toldORB_7=t6522ORB_7;
 toldIRB_7=t6522IRB_7;
 toldIRB_7L=t6522IRB_7L;
 toldORA=t6522ORA;
 toldIRA=t6522IRA;
 toldIRAL=t6522IRAL;
 toldDDRA=t6522DDRA;
 toldDDRB=t6522DDRB;
 toldSHFTREG=t6522SHFTREG;
 toldAUX_T1CTRL=t6522AUX_T1CTRL;
 toldAUX_T2CTRL=t6522AUX_T2CTRL;
 toldAUX_SHFTCTRL=t6522AUX_SHFTCTRL;
 toldAUX_PA_LE=t6522AUX_PA_LE;
 toldAUX_PB_LE=t6522AUX_PB_LE;
 toldPCR_CB1CTRL=t6522PCR_CB1CTRL;
 toldPCR_CB2CTRL=t6522PCR_CB2CTRL;
 toldPCR_CA1CTRL=t6522PCR_CA1CTRL;
 toldPCR_CA2CTRL=t6522PCR_CA2CTRL;
 toldIER=t6522IER;
 toldIFLG=t6522IFLG;
 toldCA1=t6522CA1;
 toldCA2=t6522CA2;
 toldCB1=t6522CB1;
 toldCB2=t6522CB2;
 uoldTimer1uLatchtW=u6522Timer1.uLatch.tW;
 uoldTimer1uCounttW=u6522Timer1.uCount.tW;
 uoldTimer2uLatchtW=u6522Timer2.uLatch.tW;
 uoldTimer2uCounttW=u6522Timer2.uCount.tW;
#ifdef PATCH_OFFSET
 oldwait_ramp=wait_ramp;
#endif
 if (_6522wininit==T_TRUE)
 {
  if (get_window(MonitorWinId))
   aktivate_window(get_window(MonitorWinId));
  _6522wininit=T_FALSE;
 }
 return;
}

T_INT open_6522_window(T_VOID)
{
 volatile static T_UCHAR _0;
 volatile static T_UCHAR _1;
 volatile static T_UCHAR _2;
 volatile static T_UCHAR _3;
 volatile static T_UCHAR _4;
 volatile static T_UCHAR _5;
 volatile static T_UCHAR _6;
 volatile static T_UCHAR _7;
 volatile static T_UCHAR _8;
 volatile static T_UCHAR _9;
 volatile static T_UCHAR _10;
 volatile static T_UCHAR _11;
 volatile static T_UCHAR _12;
 volatile static T_UCHAR _13;
 volatile static T_UCHAR _14;
 volatile static T_UCHAR _15;
 OPEN_WIN w;
 T_INT old_print_modus;
 T_INT old_text;
 T_INT old_back_color;
 T_INT old_text_color;
 if (get_window(_6522WinId))
  return T_TRUE;
 w.x_pos=0;
 w.y_pos=0;
 w.breite=37*8+4;
 w.hoehe=19*18;
 w.border=2;
 w.color=ddark_grau;
 w.bcolor=dhell_grau;
 w.art=RAW_WINDOW|WITH_CLOSE_BUTTON|WITH_HIDE_BUTTON|WITH_DRAG_BUTTON|REMEMBER;//|VIRTUAL_FRONT_WINDOW;
 w.id=WIN_OPEN_CHOOSES_ID;
 w.variablen=NULL;
 w.mem_bild=NULL;
 w.pcx_bild=NULL;
 w.name="6522";
 w.destructor=_6522_destructor;
 w.area_funktion=execute_6522_help;
 w.tasten_funktion=_6522_tasten;
 MOUSE_VERSTECKE
 _6522Win=open_window(&w);
 if (_6522Win==NULL)
 {
  show_message("6522 Window opening error!");
  MOUSE_ZEIGE
  return T_FALSE;
 }
 _6522WinId=_6522Win->id;
 print_srect(_6522Win,PRINT_WIN_RAW,0,0,_6522Win->b,_6522Win->h,ddark_grau,weiss,2);

 if (get_window(MonitorWinId))
 {
  remove_update_funktion(monitor_update_function);
  insert_update_funktion(_6522_update_function,1);
  monitor_update=insert_update_funktion(monitor_update_function,1);
 }
 else
  insert_update_funktion(_6522_update_function,1);
 old_print_modus=vesa_print_modus;
 old_text=vesa_text;
 old_back_color=vesa_back_color;
 old_text_color=vesa_text_color;
 vesa_print_modus=PIC_FILLED;
 vesa_text=SINGLE_COLOR;
 vesa_back_color=dhell_grau;
 vesa_text_color=ddark_grau;

 wprint(_6522Win,2,  2+0*18, "ORB_0:    IRB_0:    IRB_0L:    0:    ");
 wprint(_6522Win,2,  2+1*18, "ORB_1:    IRB_1:    IRB_1L:    1:    ");
 wprint(_6522Win,2,  2+2*18, "ORB_2:    IRB_2:    IRB_2L:    2:    ");
 wprint(_6522Win,2,  2+3*18, "ORB_3:    IRB_3:    IRB_3L:    3:    ");
 wprint(_6522Win,2,  2+4*18, "ORB_4:    IRB_4:    IRB_4L:    4:    ");
 wprint(_6522Win,2,  2+5*18, "ORB_5:    IRB_5:    IRB_5L:    5:    ");
 wprint(_6522Win,2,  2+6*18, "ORB_6:    IRB_6:    IRB_6L:    6:    ");
 wprint(_6522Win,2,  2+7*18, "ORB_7:    IRB_7:    IRB_7L:    7:    ");
 wprint(_6522Win,2,  2+8*18,   "ORA  :    IRA  :    IRAL  :    8:    ");
 if (wGblDisplayMode<5)
 {
  wprint(_6522Win,2,   2+9*18, "DDRA :    DDRB :    SHFT  :    9:    ");
  wprint(_6522Win,2,  2+10*18, "AT1C :    AT2C :    ASHFTC:    a:    ");
  wprint(_6522Win,2,  2+11*18, "APALE:    APBLE:               b:    ");
  wprint(_6522Win,2,  2+12*18, "PCRB1:    PCRB2:               c:    ");
  wprint(_6522Win,2,  2+13*18, "PCRA1:    PCRA2:               d:    ");
  wprint(_6522Win,2,  2+14*18, "IE   :    I    :               e:    ");
  wprint(_6522Win,2,  2+15*18, "CA1  :    CA2  :               f:    ");
  wprint(_6522Win,2,  2+16*18, "CB1  :    CB2  :");
  wprint(_6522Win,2,  2+17*18, "T1L  :      T1C:");
  wprint(_6522Win,2,  2+18*18, "T2L  :      T2C:");
 }
 _0=
 ((t6522ORB_0 & TMPL_RB_0)|
  (t6522ORB_1 & TMPL_RB_1)|
  (t6522ORB_2 & TMPL_RB_2)|
  (t6522ORB_3 & TMPL_RB_3)|
  (t6522ORB_4 & TMPL_RB_4)|
  (t6522ORB_5 & TMPL_RB_5)|
  (t6522ORB_6 & TMPL_RB_6)|
  (t6522ORB_7 & TMPL_RB_7));

 _1=(t6522ORA&t6522DDRA)|((t6522AUX_PA_LE)?(t6522IRAL&(t6522DDRA^0xff)):(t6522IRA&(t6522DDRA^0xff)));
 _2=(t6522DDRB);
 _3=(t6522DDRA);
 _4=(u6522Timer1.uCount.sW.tL);
 _5=(u6522Timer1.uCount.sW.tH);
 _6=(u6522Timer1.uLatch.sW.tL);
 _7=(u6522Timer1.uLatch.sW.tH);
 _8=(u6522Timer2.uCount.sW.tL);
 _9=(u6522Timer2.uCount.sW.tH);
 _10=(t6522SHFTREG);
 _11=(t6522AUX_T1CTRL<<6) | (t6522AUX_T2CTRL<<5) | (t6522AUX_SHFTCTRL<<2) | (t6522AUX_PB_LE<<1) | t6522AUX_PA_LE;;
 _12=(t6522PCR_CB2CTRL<<5) | (t6522PCR_CB1CTRL<<4) | (t6522PCR_CA2CTRL<<1) | t6522PCR_CA1CTRL;
 _13=((t6522IFLG&t6522IER)?0x80:0)|(t6522IFLG);
 _14=(0x80|t6522IER);
 _15=(t6522ORA&t6522DDRA)|(t6522IRA&(t6522DDRA^0xff));;
 print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,       2+ 0*18+_6522Win->y+_6522Win->y_0,PMAX,"$%02X IRB_0:$%02X IRB_0L:$%02X 0:$%02X",t6522ORB_0,t6522IRB_0,t6522IRB_0L,_0);
 print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,       2+ 1*18+_6522Win->y+_6522Win->y_0,PMAX,"$%02X IRB_1:$%02X IRB_1L:$%02X 1:$%02X",t6522ORB_1,t6522IRB_1,t6522IRB_1L,_1);
 print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,       2+ 2*18+_6522Win->y+_6522Win->y_0,PMAX,"$%02X IRB_2:$%02X IRB_2L:$%02X 2:$%02X",t6522ORB_2,t6522IRB_2,t6522IRB_2L,_2);
 print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,       2+ 3*18+_6522Win->y+_6522Win->y_0,PMAX,"$%02X IRB_3:$%02X IRB_3L:$%02X 3:$%02X",t6522ORB_3,t6522IRB_3,t6522IRB_3L,_3);
 print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,       2+ 4*18+_6522Win->y+_6522Win->y_0,PMAX,"$%02X IRB_4:$%02X IRB_4L:$%02X 4:$%02X",t6522ORB_4,t6522IRB_4,t6522IRB_4L,_4);
 print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,       2+ 5*18+_6522Win->y+_6522Win->y_0,PMAX,"$%02X IRB_5:$%02X IRB_5L:$%02X 5:$%02X",t6522ORB_5,t6522IRB_5,t6522IRB_5L,_5);
 print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,       2+ 6*18+_6522Win->y+_6522Win->y_0,PMAX,"$%02X IRB_6:$%02X IRB_6L:$%02X 6:$%02X",t6522ORB_6,t6522IRB_6,t6522IRB_6L,_6);
 print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,       2+ 7*18+_6522Win->y+_6522Win->y_0,PMAX,"$%02X IRB_7:$%02X IRB_7L:$%02X 7:$%02X",t6522ORB_7,t6522IRB_7,t6522IRB_7L,_7);
 print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,       2+ 8*18+_6522Win->y+_6522Win->y_0,PMAX,"$%02X IRA  :$%02X IRAL  :$%02X 8:$%02X",t6522ORA,t6522IRA,t6522IRAL,_8);

 if (wGblDisplayMode<5)
 {
  print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,   2+ 9*18+_6522Win->y+_6522Win->y_0,PMAX,"$%02X DDRB :$%02X SHFT  :$%02X 9:$%02X",t6522DDRA,t6522DDRB,t6522SHFTREG,_9);
  print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,   2+10*18+_6522Win->y+_6522Win->y_0,PMAX,"$%02X AT2C :$%02X ASHFTC:$%02X a:$%02X",t6522AUX_T1CTRL,t6522AUX_T2CTRL,t6522AUX_SHFTCTRL,_10);
  print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,   2+11*18+_6522Win->y+_6522Win->y_0,PMAX,"$%02X APBLE:$%02X            b:$%02X",t6522AUX_PA_LE,t6522AUX_PB_LE,_11);
  print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,   2+12*18+_6522Win->y+_6522Win->y_0,PMAX,"$%02X PCRB2:$%02X            c:$%02X",t6522PCR_CB1CTRL,t6522PCR_CB2CTRL,_12);
  print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,   2+13*18+_6522Win->y+_6522Win->y_0,PMAX,"$%02X PCRA2:$%02X            d:$%02X",t6522PCR_CA1CTRL,t6522PCR_CA2CTRL,_13);
  print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,   2+14*18+_6522Win->y+_6522Win->y_0,PMAX,"$%02X I    :$%02X            e:$%02X",t6522IER,t6522IFLG,_14);
  print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,   2+15*18+_6522Win->y+_6522Win->y_0,PMAX,"$%02X CA2  :$%02X            f:$%02X",t6522CA1,t6522CA2,_15);
  print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,   2+16*18+_6522Win->y+_6522Win->y_0,PMAX,"$%02X CB2  :$%02X",t6522CB1,t6522CB2);
  print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,   2+17*18+_6522Win->y+_6522Win->y_0,PMAX,"$%04X T1C:$%04X",u6522Timer1.uLatch.tW,u6522Timer1.uCount.tW);
  print_with_all_features(_6522Win,PRINT_RAW_OC,2+6*8+_6522Win->x+_6522Win->x_0,   2+18*18+_6522Win->y+_6522Win->y_0,PMAX,"$%04X T2C:$%04X",u6522Timer2.uLatch.tW,u6522Timer2.uCount.tW);
#ifdef PATCH_OFFSET
 vesa_text_color=f_red;
  if (wait_ramp==T_TRUE)
   print_with_all_features(_6522Win,PRINT_CLIPPED_OC,2+24*8 , 2+18*18 ,PMAX,"PATCHING");
#endif
 }
 update_whole_window(_6522Win);
 vesa_print_modus=old_print_modus;
 vesa_text=old_text;
 vesa_back_color=old_back_color;
 vesa_text_color=old_text_color;
 MOUSE_ZEIGE
 return T_TRUE;
}

T_UCHAR f6522Win(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[])
{
 _6522wininit=T_TRUE;
 open_6522_window();
 return 0;
}

T_VOID execute_6522_help(PARAMETER *parameter)
{
 WINDOW_LISTEN_ELEMENT *window=(WINDOW_LISTEN_ELEMENT *)parameter->window_element;
 if (parameter==NULL)
 {
  return;
 }
 if (window==NULL)
 {
  return;
 }
 if (parameter->maus_button==1)
 {
  parameter->executed=PARAMETER_EXECUTED;
  aktivate_window(window);
 }
 if (parameter->maus_button==2)
 {
  /* set cursor to this position */
//  T_INT maus_y=parameter->maus_y-(window->y+window->y_0);
//  maus_y+=(font_now->zeichen[' '].hoehe-1);
//  maus_y/=(font_now->zeichen[' '].hoehe);
  T_INT maus_y=parameter->maus_y-(window->y+window->y_0);
  T_INT maus_x=parameter->maus_x-(window->x+window->x_0);
  parameter->executed=PARAMETER_EXECUTED;
  maus_y+=17;
  maus_y/=18;
  switch (maus_y)
  {
   case 1:
   {
    if (maus_x<250)
     help_function("6522\\ORB_0.hlf",NULL);
    else
     help_function("6522\\d000.hlf",NULL);
    break;
   }
   case 2:
   {
    if (maus_x<250)
     help_function("6522\\ORB_1.hlf",NULL);
    else
     help_function("6522\\d001.hlf",NULL);
    break;
   }
   case 3:
   {
    if (maus_x<250)
     help_function("6522\\ORB_2.hlf",NULL);
    else
     help_function("6522\\d002.hlf",NULL);
   break;
   }
   case 4:
   {
    if (maus_x<250)
     help_function("6522\\ORB_3.hlf",NULL);
    else
     help_function("6522\\d003.hlf",NULL);
    break;
   }
   case 5:
   {
    if (maus_x<250)
     help_function("6522\\ORB_4.hlf",NULL);
    else
     help_function("6522\\d004.hlf",NULL);
   break;
   }
   case 6:
   {
    if (maus_x<250)
     help_function("6522\\ORB_5.hlf",NULL);
    else
     help_function("6522\\d005.hlf",NULL);
    break;
   }
   case 7:
   {
    if (maus_x<250)
     help_function("6522\\ORB_6.hlf",NULL);
    else
     help_function("6522\\d006.hlf",NULL);
    break;
   }
   case 8:
   {
    if (maus_x<250)
     help_function("6522\\ORB_7.hlf",NULL);
    else
     help_function("6522\\d007.hlf",NULL);
    break;
   }
   case 9:
   {
    if (maus_x<250)
     help_function("6522\\ORA.hlf",NULL);
    else
     help_function("6522\\d008.hlf",NULL);
    break;
   }
   case 10:
   {
    if (maus_x<80)
     help_function("6522\\DDRA.hlf",NULL);
    else if (maus_x<160)
     help_function("6522\\DDRB.hlf",NULL);
    else if (maus_x<250)
     help_function("6522\\SHFT.hlf",NULL);
    else
     help_function("6522\\d009.hlf",NULL);
    break;
   }
   case 11:
   {
    if (maus_x<80)
     help_function("6522\\AT1C.hlf",NULL);
    else if (maus_x<160)
     help_function("6522\\AT2C.hlf",NULL);
    else if (maus_x<250)
     help_function("6522\\ASHFTC.hlf",NULL);
    else
     help_function("6522\\d00a.hlf",NULL);
    break;
   }
   case 12:
   {
    if (maus_x<80)
     help_function("6522\\APALE.hlf",NULL);
    else if (maus_x<160)
     help_function("6522\\APBLE.hlf",NULL);
    else
     help_function("6522\\d00b.hlf",NULL);
    break;
   }
   case 13:
   {
    if (maus_x<80)
     help_function("6522\\PCRB1.hlf",NULL);
    else if (maus_x<160)
     help_function("6522\\PCRB2.hlf",NULL);
    else
     help_function("6522\\d00c.hlf",NULL);
    break;
   }
   case 14:
   {
    if (maus_x<80)
     help_function("6522\\PCRA1.hlf",NULL);
    else if (maus_x<160)
     help_function("6522\\PCRA2.hlf",NULL);
    else
     help_function("6522\\d00d.hlf",NULL);
    break;
   }
   case 15:
   {
    if (maus_x<80)
     help_function("6522\\IE.hlf",NULL);
    else if (maus_x<160)
     help_function("6522\\I.hlf",NULL);
    else
     help_function("6522\\d00e.hlf",NULL);
    break;
   }
   case 16:
   {
    if (maus_x<80)
     help_function("6522\\CA1.hlf",NULL);
    else if (maus_x<160)
     help_function("6522\\CA2.hlf",NULL);
    else
     help_function("6522\\d00f.hlf",NULL);
    break;
   }
   case 17:
   {
    if (maus_x<80)
     help_function("6522\\CB1.hlf",NULL);
    else if (maus_x<160)
     help_function("6522\\CB2.hlf",NULL);
    break;
   }
   case 18:
   {
    help_function("6522\\T1.hlf",NULL);
    break;
   }
   case 19:
   {
    help_function("6522\\T2.hlf",NULL);
    break;
   }
  }
 }
}
/***************************************************************/
/***** PSG Information Window **********************************/
/***************************************************************/
/* Only default.font supported 8*16 */
/************************************/

T_INT PSG_tasten(PARAMETER *parameter)
{
 if (get_window(MonitorWinId))
  aktivate_window(get_window(MonitorWinId));
 if (bStep==T_TRUE)
 {
  steps=1;
 }
 return 0;
}

T_INT PSG_destructor(PARAMETER *parameter)
{
 WINDOW_LISTEN_ELEMENT *window=(WINDOW_LISTEN_ELEMENT *)parameter->window_element;
 if (parameter==NULL)
 {
  show_message("No Parameter in PSG_destructor()");
  return -1;
 }
 if (window==NULL)
 {
  show_message("I can't find myself!!! (PSG Window not closed)");
  return -1;
 }
// interesting part of destructor (just a notifier in this case)
 remove_update_funktion(PSG_update_function);
 window->destructor=NULL; // the rest is done by close_window
 PSGWinId=-1;
 PSGWin=NULL;
 return -1; // keine variablen da, dann ebene = -1
}

T_VOID PSG_update_function(PARAMETER *parameter)
{
 static T_LONG old_pos=-1;
 T_INT old_print_modus;
 T_INT old_back_color;
 T_INT old_text_color;
 T_INT old_text;
 if (old_pos==lwGblSystemTickTotal)
  return;
 old_pos=lwGblSystemTickTotal;
 if (PSGWin->hide!=0)
  return;

 old_print_modus=vesa_print_modus;
 old_text=vesa_text;
 old_back_color=vesa_back_color;
 old_text_color=vesa_text_color;
 vesa_print_modus=PIC_FILLED;
 vesa_text=SINGLE_COLOR;
 vesa_back_color=dhell_grau;
 vesa_text_color=ddark_grau;

 t_hide_win_mouse(PSGWin);
 if (wGblSound==SOUND_ADLIB)
 {
  T_INT i;
  if (((tOldsamp!=tSIGDEF_SAMPHOLD)||(PSG_Reg[0]!=old_psg_reg[0]))||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 1*18 ,PMAX,"%03i       SIGDEF_SAMPHOLD :%03i",PSG_Reg[0],tSIGDEF_SAMPHOLD);
  if (((tOldamux!=tSIGDEF_AMUXSEL)||(PSG_Reg[1]!=old_psg_reg[1]))||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 2*18 ,PMAX,"%03i       SIGDEF_AMUXSEL  :%03i",PSG_Reg[1],tSIGDEF_AMUXSEL);
  if (((tOldaybc!=tSIGDEF_AY3BC1)||(PSG_Reg[3]!=old_psg_reg[3]))||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 3*18 ,PMAX,"%03i       SIGDEF_AY3BC1   :%03i",PSG_Reg[3],tSIGDEF_AY3BC1);
  if (((tOldaydir!=tSIGDEF_AY3BDIR)||(PSG_Reg[2]!=old_psg_reg[2]))||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 4*18 ,PMAX,"%03i       SIGDEF_AY3BDIR  :%03i",PSG_Reg[2],tSIGDEF_AY3BDIR);
  if (((tOldOra_!=t6522ORA)||(PSG_Reg[4]!=old_psg_reg[4]))||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 5*18 ,PMAX,"%03i       ORA             :%03i",PSG_Reg[4],t6522ORA);
  if (((tOldIra_!=t6522IRA)||(PSG_Reg[5]!=old_psg_reg[5]))||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 6*18 ,PMAX,"%03i       IRA             :%03i",PSG_Reg[5],t6522IRA);
  if ((PSG_Reg[6]!=old_psg_reg[6])||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 7*18 ,PMAX,"%03i",PSG_Reg[6]);
  if ((PSG_Reg[7]!=old_psg_reg[7])||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 8*18 ,PMAX,"%03i",PSG_Reg[7]);
  if (wGblDisplayMode<5)
  {
   if ((PSG_Reg[8]!=old_psg_reg[8])||(parameter==1))
    print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 9*18 ,PMAX,"%03i",PSG_Reg[8]);
   if ((PSG_Reg[9]!=old_psg_reg[9])||(parameter==1))
    print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+10*18 ,PMAX,"%03i",PSG_Reg[9]);
   if ((PSG_Reg[10]!=old_psg_reg[10])||(parameter==1))
    print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+11*18 ,PMAX,"%03i",PSG_Reg[10]);
   if ((PSG_Reg[11]!=old_psg_reg[11])||(parameter==1))
    print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+12*18 ,PMAX,"%03i",PSG_Reg[11]);
   if ((PSG_Reg[12]!=old_psg_reg[12])||(parameter==1))
    print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+13*18 ,PMAX,"%03i",PSG_Reg[12]);
   if ((PSG_Reg[13]!=old_psg_reg[13])||(parameter==1))
    print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+14*18 ,PMAX,"%03i",PSG_Reg[13]);
  } // if (wGblDisplayMode<5)
  for (i=0;i<14;i++)
   old_psg_reg[i]=PSG_Reg[i];
 } // if (wGblSound==SOUND_ADLIB)
 if (wGblSound==SOUND_SEAL)
 {
  T_INT i;
  if (((tOldsamp!=tSIGDEF_SAMPHOLD)||(Regs[0]!=old_psg_reg[0]))||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 1*18 ,PMAX,"%03i       SIGDEF_SAMPHOLD :%03i",Regs[0],tSIGDEF_SAMPHOLD);
  if (((tOldamux!=tSIGDEF_AMUXSEL)||(Regs[1]!=old_psg_reg[1]))||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 2*18 ,PMAX,"%03i       SIGDEF_AMUXSEL  :%03i",Regs[1],tSIGDEF_AMUXSEL);
  if (((tOldaybc!=tSIGDEF_AY3BC1)||(Regs[3]!=old_psg_reg[3]))||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 3*18 ,PMAX,"%03i       SIGDEF_AY3BC1   :%03i",Regs[3],tSIGDEF_AY3BC1);
  if (((tOldaydir!=tSIGDEF_AY3BDIR)||(Regs[2]!=old_psg_reg[2]))||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 4*18 ,PMAX,"%03i       SIGDEF_AY3BDIR  :%03i",Regs[2],tSIGDEF_AY3BDIR);
  if (((tOldOra_!=t6522ORA)||(Regs[4]!=old_psg_reg[4]))||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 5*18 ,PMAX,"%03i       ORA             :%03i",Regs[4],t6522ORA);
  if (((tOldIra_!=t6522IRA)||(Regs[5]!=old_psg_reg[5]))||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 6*18 ,PMAX,"%03i       IRA             :%03i",Regs[5],t6522IRA);
  if ((Regs[6]!=old_psg_reg[6])||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 7*18 ,PMAX,"%03i",Regs[6]);
  if ((Regs[7]!=old_psg_reg[7])||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 8*18 ,PMAX,"%03i",Regs[7]);
  if (wGblDisplayMode<5)
  {
   if ((Regs[8]!=old_psg_reg[8])||(parameter==1))
    print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 9*18 ,PMAX,"%03i",Regs[8]);
   if ((Regs[9]!=old_psg_reg[9])||(parameter==1))
    print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+10*18 ,PMAX,"%03i",Regs[9]);
   if ((Regs[10]!=old_psg_reg[10])||(parameter==1))
    print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+11*18 ,PMAX,"%03i",Regs[10]);
   if ((Regs[11]!=old_psg_reg[11])||(parameter==1))
    print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+12*18 ,PMAX,"%03i",Regs[11]);
   if ((Regs[12]!=old_psg_reg[12])||(parameter==1))
    print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+13*18 ,PMAX,"%03i",Regs[12]);
   if ((Regs[13]!=old_psg_reg[13])||(parameter==1))
    print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+14*18 ,PMAX,"%03i",Regs[13]);
  } // if (wGblDisplayMode<5)
  for (i=0;i<14;i++)
   old_psg_reg[i]=Regs[i];
 } // if (wGblSound==SOUND_SEAL)
 if (wGblSound==SOUND_NONE)
 {
  if ((tOldsamp!=tSIGDEF_SAMPHOLD)||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 1*18 ,PMAX,"N/A       SIGDEF_SAMPHOLD :%03i",tSIGDEF_SAMPHOLD);
  if ((tOldamux!=tSIGDEF_AMUXSEL)||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 2*18 ,PMAX,"N/A       SIGDEF_AMUXSEL  :%03i",tSIGDEF_AMUXSEL);
  if ((tOldaybc!=tSIGDEF_AY3BC1)||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 3*18 ,PMAX,"N/A       SIGDEF_AY3BC1   :%03i",tSIGDEF_AY3BC1);
  if ((tOldaydir!=tSIGDEF_AY3BDIR)||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 4*18 ,PMAX,"N/A       SIGDEF_AY3BDIR  :%03i",tSIGDEF_AY3BDIR);
  if ((tOldOra_!=t6522ORA)||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 5*18 ,PMAX,"N/A       ORA             :%03i",t6522ORA);
  if ((tOldIra_!=t6522IRA)||(parameter==1))
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(2)*8 ,   2+ 5*18 ,PMAX,"N/A       IRA             :%03i",t6522IRA);
 } // if (wGblSound==SOUND_NONE)
 if (((tOldKeyOut!=tKeyOut)||(tOldKeyIn!=tKeyWord))||(parameter==1))
 {
  print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(19)*8,2+ 7*18,PMAX,"%03i %03i",tKeyWord,tKeyOut);
  print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(15)*8,2+ 8*18,PMAX,"%03i %03i",tKeyWord&1,tKeyOut&1);
  if (wGblDisplayMode<5)
  {
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(15)*8,2+ 9*18 ,PMAX,"%03i %03i",tKeyWord&2,tKeyOut&2);
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(15)*8,2+ 10*18,PMAX,"%03i %03i",tKeyWord&4,tKeyOut&4);
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(15)*8,2+ 11*18,PMAX,"%03i %03i",tKeyWord&8,tKeyOut&8);
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(15)*8,2+ 12*18,PMAX,"%03i %03i",tKeyWord&16,tKeyOut&16);
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(15)*8,2+ 13*18,PMAX,"%03i %03i",tKeyWord&32,tKeyOut&32);
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(15)*8,2+ 14*18,PMAX,"%03i %03i",tKeyWord&64,tKeyOut&64);
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(15)*8,2+ 15*18,PMAX,"%03i %03i",tKeyWord&128,tKeyOut&128);
  }
 }
 if ((imager_pulse_ticks!=oldimager_pulse_ticks)||(parameter==1))
 {
  if (wGblDisplayMode<5)
  {
   print_with_all_features(PSGWin,PRINT_CLIPPED_OC,2+(19)*8,2+ 16*18,PMAX,"%i      ",imager_pulse_ticks);
  }
 }



 tOldsamp=tSIGDEF_SAMPHOLD;
 tOldamux=tSIGDEF_AMUXSEL;
 tOldaydir=tSIGDEF_AY3BDIR;
 tOldaybc=tSIGDEF_AY3BC1;
 tOldOra_=t6522ORA;
 tOldIra_=t6522IRA;
 tOldKeyIn=tKeyWord;
 tOldKeyOut=tKeyOut;
 oldimager_pulse_ticks=imager_pulse_ticks;
 t_show_win_mouse(PSGWin);
 vesa_print_modus=old_print_modus;
 vesa_text=old_text;
 vesa_back_color=old_back_color;
 vesa_text_color=old_text_color;

 if (PSGwininit==T_TRUE)
 {
  if (get_window(MonitorWinId))
   aktivate_window(get_window(MonitorWinId));
  PSGwininit=T_FALSE;
 }
 return;
}

T_INT open_PSG_window(T_VOID)
{
 OPEN_WIN w;
 T_INT old_print_modus;
 T_INT old_text;
 T_INT old_back_color;
 T_INT old_text_color;
 if (get_window(PSGWinId))
  return T_TRUE;
 w.x_pos=0;
 w.y_pos=0;
 w.breite=32*8+4;
 w.hoehe=17*18;
 w.border=2;
 w.color=ddark_grau;
 w.bcolor=dhell_grau;
 w.art=RAW_WINDOW|WITH_CLOSE_BUTTON|WITH_HIDE_BUTTON|WITH_DRAG_BUTTON|REMEMBER;//|VIRTUAL_FRONT_WINDOW;
 w.id=WIN_OPEN_CHOOSES_ID;
 w.variablen=NULL;
 w.mem_bild=NULL;
 w.pcx_bild=NULL;
 w.name="AY-3-8192";
 w.destructor=PSG_destructor;
 w.area_funktion=execute_PSG_help;
 w.tasten_funktion=PSG_tasten;
 MOUSE_VERSTECKE
 PSGWin=open_window(&w);
 if (PSGWin==NULL)
 {
  show_message("PSG Window opening error!");
  MOUSE_ZEIGE
  return T_FALSE;
 }
 PSGWinId=PSGWin->id;
 print_srect(PSGWin,PRINT_WIN_RAW,0,0,PSGWin->b,PSGWin->h,ddark_grau,weiss,2);

 if (get_window(MonitorWinId))
 {
  remove_update_funktion(monitor_update_function);
  insert_update_funktion(PSG_update_function,1);
  monitor_update=insert_update_funktion(monitor_update_function,1);
 }
 else
  insert_update_funktion(PSG_update_function,1);
 old_print_modus=vesa_print_modus;
 old_text=vesa_text;
 old_back_color=vesa_back_color;
 old_text_color=vesa_text_color;
 vesa_print_modus=PIC_FILLED;
 vesa_text=SINGLE_COLOR;
 vesa_back_color=dhell_grau;
 vesa_text_color=ddark_grau;

 wprint(PSGWin,2,  2+0*18, "AY-3-8192   6522 relevant");
 wprint(PSGWin,2,  2+1*18, "0:N/A       SIGDEF_SAMPHOLD :");
 wprint(PSGWin,2,  2+2*18, "1:N/A       SIGDEF_AMUXSEL  :");
 wprint(PSGWin,2,  2+3*18, "3:N/A       SIGDEF_AY3BC1   :");
 wprint(PSGWin,2,  2+4*18, "2:N/A       SIGDEF_AY3BDIR  :");
 wprint(PSGWin,2,  2+5*18, "4:N/A       ORA             :");
 wprint(PSGWin,2,  2+6*18, "5:N/A       IRA             :");
 wprint(PSGWin,2,  2+7*18, "6:N/A       I/O (e:       )");
 wprint(PSGWin,2,  2+8*18, "7:N/A       #0:");
 if (wGblDisplayMode<5)
 {
  wprint(PSGWin,2,  2+9*18, "8:N/A       #1:");
  wprint(PSGWin,2,  2+10*18, "9:N/A       #2:");
  wprint(PSGWin,2,  2+11*18, "a:N/A       #3:");
  wprint(PSGWin,2,  2+12*18, "b:N/A       #4:");
  wprint(PSGWin,2,  2+13*18, "c:N/A       #5:");
  wprint(PSGWin,2,  2+14*18, "d:N/A       #6:");
  wprint(PSGWin,2,  2+15*18, "            #7:");
  wprint(PSGWin,2,  2+16*18, "Imager pulse ticks:");
 }


 print_with_all_features(PSGWin,PRINT_RAW_OC,2+(15)*8+PSGWin->x+PSGWin->x_0,2+ 8*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i 03%i",tKeyWord&1,tKeyOut&1);
 if (wGblDisplayMode<5)
 {
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(15)*8+PSGWin->x+PSGWin->x_0,2+ 9*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i %03i",tKeyWord&2,tKeyOut&2);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(15)*8+PSGWin->x+PSGWin->x_0,2+ 10*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i %03i",tKeyWord&4,tKeyOut&4);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(15)*8+PSGWin->x+PSGWin->x_0,2+ 11*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i %03i",tKeyWord&8,tKeyOut&8);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(15)*8+PSGWin->x+PSGWin->x_0,2+ 12*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i %03i",tKeyWord&16,tKeyOut&16);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(15)*8+PSGWin->x+PSGWin->x_0,2+ 13*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i %03i",tKeyWord&32,tKeyOut&32);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(15)*8+PSGWin->x+PSGWin->x_0,2+ 14*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i %03i",tKeyWord&64,tKeyOut&64);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(15)*8+PSGWin->x+PSGWin->x_0,2+ 15*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i %03i",tKeyWord&128,tKeyOut&128);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(19)*8+PSGWin->x+PSGWin->x_0,2+ 15*18+PSGWin->y+PSGWin->y_0,PMAX,"%i      ",imager_pulse_ticks);
 }

 if (wGblSound==SOUND_ADLIB)
 {
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 1*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i       SIGDEF_SAMPHOLD :%03i",PSG_Reg[0],tSIGDEF_SAMPHOLD);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 2*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i       SIGDEF_AMUXSEL  :%03i",PSG_Reg[1],tSIGDEF_AMUXSEL);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 3*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i       SIGDEF_AY3BC1   :%03i",PSG_Reg[3],tSIGDEF_AY3BC1);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 4*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i       SIGDEF_AY3BDIR  :%03i",PSG_Reg[2],tSIGDEF_AY3BDIR);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 5*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i       ORA             :%03i",PSG_Reg[4],t6522ORA);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 6*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i       IRA             :%03i",PSG_Reg[5],t6522IRA);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 7*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i       I/O (e:%03i %03i",PSG_Reg[6],tKeyWord,tKeyOut);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 8*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i",PSG_Reg[7]);
  if (wGblDisplayMode<5)
  {
   print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 9*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i",PSG_Reg[8]);
   print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+10*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i",PSG_Reg[9]);
   print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+11*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i",PSG_Reg[10]);
   print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+12*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i",PSG_Reg[11]);
   print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+13*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i",PSG_Reg[12]);
   print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+14*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i",PSG_Reg[13]);
  } // if (wGblDisplayMode<5)
 } // if (wGblSound==SOUND_ADLIB)
 if (wGblSound==SOUND_SEAL)
 {
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 1*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i       SIGDEF_SAMPHOLD :%03i",Regs[0],tSIGDEF_SAMPHOLD);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 2*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i       SIGDEF_AMUXSEL  :%03i",Regs[1],tSIGDEF_AMUXSEL);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 3*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i       SIGDEF_AY3BC1   :%03i",Regs[3],tSIGDEF_AY3BC1);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 4*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i       SIGDEF_AY3BDIR  :%03i",Regs[2],tSIGDEF_AY3BDIR);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 5*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i       ORA             :%03i",Regs[4],t6522ORA);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 6*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i       IRA             :%03i",Regs[5],t6522IRA);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 7*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i       I/O (e:%03i %03i",Regs[6],tKeyWord,tKeyOut);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 8*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i",Regs[7]);
  if (wGblDisplayMode<5)
  {
   print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 9*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i",Regs[8]);
   print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+10*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i",Regs[9]);
   print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+11*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i",Regs[10]);
   print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+12*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i",Regs[11]);
   print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+13*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i",Regs[12]);
   print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+14*18+PSGWin->y+PSGWin->y_0,PMAX,"%03i",Regs[13]);
  } // if (wGblDisplayMode<5)
 } // if (wGblSound==SOUND_SEAL)
 if (wGblSound==SOUND_NONE)
 {
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 1*18+PSGWin->y+PSGWin->y_0,PMAX,"N/A       SIGDEF_SAMPHOLD :%03i",tSIGDEF_SAMPHOLD);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 2*18+PSGWin->y+PSGWin->y_0,PMAX,"N/A       SIGDEF_AMUXSEL  :%03i",tSIGDEF_AMUXSEL);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 3*18+PSGWin->y+PSGWin->y_0,PMAX,"N/A       SIGDEF_AY3BC1   :%03i",tSIGDEF_AY3BC1);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 4*18+PSGWin->y+PSGWin->y_0,PMAX,"N/A       SIGDEF_AY3BDIR  :%03i",tSIGDEF_AY3BDIR);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 5*18+PSGWin->y+PSGWin->y_0,PMAX,"N/A       ORA             :%03i",t6522ORA);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 5*18+PSGWin->y+PSGWin->y_0,PMAX,"N/A       IRA             :%03i",t6522IRA);
  print_with_all_features(PSGWin,PRINT_RAW_OC,2+(2)*8+PSGWin->x+PSGWin->x_0,   2+ 5*18+PSGWin->y+PSGWin->y_0,PMAX,"N/A       I/O (e:%03i %03i",tKeyWord,tKeyOut);
 } // if (wGblSound==SOUND_NONE)
 update_whole_window(PSGWin);
 vesa_print_modus=old_print_modus;
 vesa_text=old_text;
 vesa_back_color=old_back_color;
 vesa_text_color=old_text_color;
 MOUSE_ZEIGE
 return T_TRUE;
}

T_UCHAR fPSGWin(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[])
{
 PSGwininit=T_TRUE;
 open_PSG_window();
 return 0;
}

T_VOID execute_PSG_help(PARAMETER *parameter)
{
 WINDOW_LISTEN_ELEMENT *window=(WINDOW_LISTEN_ELEMENT *)parameter->window_element;
 if (parameter==NULL)
 {
  return;
 }
 if (window==NULL)
 {
  return;
 }
 if (parameter->maus_button==1)
 {
  parameter->executed=PARAMETER_EXECUTED;
  aktivate_window(window);
 }
 if (parameter->maus_button==2)
 {
  /* set cursor to this position */
  T_INT maus_y=parameter->maus_y-(window->y+window->y_0);
  T_INT maus_x=parameter->maus_x-(window->x+window->x_0);
  parameter->executed=PARAMETER_EXECUTED;
  maus_y+=17;
  maus_y/=18;
  switch (maus_y-1)
  {
   case 1:
   {
    if (maus_x<12*8)
     help_function("psg\\reg0.hlf",NULL);
    else
     help_function("analog\\sigshold.hlf",NULL);
    break;
   }
   case 2:
   {
    if (maus_x<12*8)
     help_function("psg\\reg1.hlf",NULL);
    else
     help_function("analog\\sigasel.hlf",NULL);
    break;
   }
   case 3:
   {
    if (maus_x<12*8)
     help_function("psg\\reg2.hlf",NULL);
    else
     help_function("psg\\sigbc1.hlf",NULL);
    break;
   }
   case 4:
   {
    if (maus_x<12*8)
     help_function("psg\\reg3.hlf",NULL);
    else
     help_function("psg\\sigbdir.hlf",NULL);
    break;
   }
   case 5:
   {
    if (maus_x<12*8)
     help_function("psg\\reg4.hlf",NULL);
    else
     help_function("6522\\ora.hlf",NULL);

    break;
   }
   case 6:
   {
    if (maus_x<12*8)
     help_function("psg\\reg5.hlf",NULL);
    else
     help_function("6522\\ora.hlf",NULL);
    break;
   }
   case 7:
   {
    if (maus_x<12*8)
     help_function("psg\\reg6.hlf",NULL);
    else
     help_function("psg\\rege.hlf",NULL);
    break;
   }
   case 8:
   {
    if (maus_x<12*8)
     help_function("psg\\reg7.hlf",NULL);
    else
     help_function("psg\\rege0.hlf",NULL);
    break;
   }
   case 9:
   {
    if (maus_x<12*8)
     help_function("psg\\reg8.hlf",NULL);
    else
     help_function("psg\\rege1.hlf",NULL);
    break;
   }
   case 10:
   {
    if (maus_x<12*8)
     help_function("psg\\reg9.hlf",NULL);
    else
     help_function("psg\\rege2.hlf",NULL);
    break;
   }
   case 11:
   {
    if (maus_x<12*8)
     help_function("psg\\rega.hlf",NULL);
    else
     help_function("psg\\rege3.hlf",NULL);
    break;
   }
   case 12:
   {
    if (maus_x<12*8)
     help_function("psg\\regb.hlf",NULL);
    else
     help_function("psg\\rege4.hlf",NULL);
    break;
   }
   case 13:
   {
    if (maus_x<12*8)
     help_function("psg\\regc.hlf",NULL);
    else
     help_function("psg\\rege5.hlf",NULL);
    break;
   }
   case 14:
   {
    if (maus_x<12*8)
     help_function("psg\\regd.hlf",NULL);
    else
     help_function("psg\\rege6.hlf",NULL);
    break;
   }
   case 15:
   {
    help_function("psg\\rege7.hlf",NULL);
    break;
   }
   case 16:
   {
    help_function("psg\\imager.hlf",NULL);
    break;
   }
  }
 }
}

T_UCHAR fDissi(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[])
{
 T_UWORD wAddress;
 T_UWORD wStart;
 T_UWORD wEnd;
 T_LISTE *liste;
 T_CHAR string[STRING_LENGTH];

 wStart=0;
 wEnd=wStart+DISAS_DEFAULT_LENGTH;
 /* Proces the input params */
 switch (iArgC)
 {
  case 3:
   if (fStringToWord(apcArgV[1],(T_UWORD *)&wStart)==T_FALSE)
   {
    edit_log_add(get_window(MonitorWinId),"Invalid Start Address.");
    edit_log_add(get_window(MonitorWinId),"");
    return(DBG_ERROR);
   }
   if (fStringToWord(apcArgV[2],(T_UWORD *)&wEnd)==T_FALSE)
   {
    edit_log_add(get_window(MonitorWinId),"Invalid End Address.");
    edit_log_add(get_window(MonitorWinId),"");
    return(DBG_ERROR);
   }
   wEnd=wStart+wEnd;
   if (wEnd<wStart)
   {
    edit_log_add(get_window(MonitorWinId),"Invalid End Address.");
    edit_log_add(get_window(MonitorWinId),"");
    return(DBG_ERROR);
   }
   break;
  case 2:
   if (fStringToWord(apcArgV[1],(T_UWORD *)&wStart)==T_FALSE)
   {
    edit_log_add(get_window(MonitorWinId),"Invalid Start Address.");
    edit_log_add(get_window(MonitorWinId),"");
    return(DBG_ERROR);
   }
   wEnd=wStart+DISAS_DEFAULT_LENGTH;
   break;
  case 1:
   break;
  default:
   edit_log_add(get_window(MonitorWinId),"Too many arguments.");
   edit_log_add(get_window(MonitorWinId),"Usage: SOURCE xxxx, yy");
   edit_log_add(get_window(MonitorWinId),"");
   return(DBG_ERROR);
 }
 wAddress=wStart;
 while (wAddress<wEnd)
 {
  if (dis_mem_map[wAddress]&D_MEM_HAS_INFO)
  {
   liste=info_liste_first;
   while (liste!=NULL)
   {
    D_ADDRESS_INFO *info;
    info=(D_ADDRESS_INFO *)(liste->objekt);
    if (info->address==wAddress)
    {
     t_printf(0,"%s\n",info->string);
    }
    liste=liste->naechstes;
   }
  }
  wAddress=f6809Disassemble(wAddress,string);
  t_printf(0,"%s\n",string);
 } // while (wAddress<wEnd)
 return(DBG_OK);
}
T_UCHAR fpause(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[])
{
 if (bGblPauseMode==T_TRUE)
  bGblPauseMode=T_FALSE;
 else
 {
  bGblPauseMode=T_TRUE;
  vector_undraw();
  vector_refresh();
 }
 return(DBG_OK);
}

T_UCHAR frefresh(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[])
{
 register_update_function((PARAMETER *)1);
 analog_update_function((PARAMETER *)1);
 _6522_update_function((PARAMETER *)1);
 PSG_update_function((PARAMETER *)1);
 return(DBG_OK);
}

T_INT monitor_interpreter(T_VOID *parameter)
{
 EDIT_ELEMENT *ed=(EDIT_ELEMENT *)parameter;
 T_LISTE *zeilen_liste;
 EDIT_ZEILE *last_line;
 T_UBYTE acCommandLine[MAX_COMMAND_LINE_LENGTH+1];
 T_PUBYTE       apcArgV[MAX_COMMAND_LINE_ARGS];
 T_UBYTE acCommandLine1[MAX_COMMAND_LINE_LENGTH+1]; // these '1' store a copy of the unuppered command line
 T_PUBYTE       apcArgV1[MAX_COMMAND_LINE_ARGS];
 T_SWORD        iArgC,iLineLength;
 T_SWORD        iTmp;
 acCommandLine[0]=(T_CHAR)0;
 acCommandLine1[0]=(T_CHAR)0;
 zeilen_liste=d_get_list_pos(ed->cursor_line, ed->text_zeile_erste);
 zeilen_liste=zeilen_liste->letztes;
 last_line=(EDIT_ZEILE *)zeilen_liste->objekt;
 strcat(acCommandLine,last_line->text);
 strcat(acCommandLine,ed->cursor_line->text);
 strcat(acCommandLine1,last_line->text);
 strcat(acCommandLine1,ed->cursor_line->text);
 fDebug(HIGH,("fDebugMonitor(): Got input line \"%s\"",acCommandLine));
 if (strlen(acCommandLine)!=0)
 {
  strcpy(last_acCommandLine[buffer_in_pos],acCommandLine);
  buffer_in_pos=(buffer_in_pos+1)%BACK_BUFFER_LENGTH;
  buffer_out_pos=buffer_in_pos;
 }
 /* Scan the command line and convert to arg list */
 iArgC=0;
 iTmp=0;
 iLineLength=strlen(acCommandLine);

 while (iTmp<iLineLength && iArgC<MAX_COMMAND_LINE_ARGS)
 {
  if (fIsWhiteSpace(acCommandLine[iTmp])==T_TRUE)
  {
   iTmp++;
  }
  else
  {
   /* Now we found a string start, register the address */
   apcArgV1[iArgC]=&acCommandLine1[iTmp];
   apcArgV[iArgC++]=&acCommandLine[iTmp];

   /* Scan to the end of the field */
   for (;;)
   {
    if (fIsWhiteSpace(acCommandLine[iTmp])==T_FALSE && acCommandLine[iTmp]!=0)
    {
     acCommandLine[iTmp]=toupper(acCommandLine[iTmp]);
     iTmp++;
    }
    else
    {
     acCommandLine[iTmp]=0;
     acCommandLine1[iTmp]=0;
     fDebug(MED,("fDebugMonitor(): argv[%d] = \"%s\"",iArgC-1,apcArgV[iArgC-1]));
     iTmp++;
     break;
    }
   }
  }
 }


 /* Do a string match for agrv0 for command name and execute the */
 /* associated command in the command list array                 */
 if (iArgC>0)
 {
  iTmp=0;
  for (;;)
  {
   if (strcmp(apcArgV[0],asCommandList[iTmp].pcCommandName)==0)
   {
    T_INT s;
    /* Now we've got a match, execute the command */
    fDebug(MED,("fDebugMonitor(): executing command %d (%s)",iTmp,asCommandList[iTmp].pcCommandName));

    /* If a command returns T_TRUE then exit debug, this is used for */
    /* single step and run commands that don't want the window closed */
    s=(*asCommandList[iTmp].pfCommandFunc)(iArgC,apcArgV,apcArgV1);
    switch (s)
    {
     case DBG_OK:
      fDebug(HIGH,("fDebugMonitor(): returned OK"));
      break;
     case DBG_ERROR:
      fDebug(HIGH,("fDebugMonitor(): returned ERROR"));
      break;
     case DBG_RUN:
      fDebug(HIGH,("fDebugMonitor(): returned RUN"));
      bGblPauseMode=T_FALSE;
      break;
     case DBG_QUIT:
      fDebug(HIGH,("fDebugMonitor(): returned QUIT"));
       close_window(get_window(MonitorWinId),CLOSE_DEFAULT);
       return EDIT_QUIT;
     case DBG_SEQUENCE:
      fDebug(HIGH,("fDebugMonitor(): returned SEQUENCE"));
      bGblPauseMode=T_FALSE;
      break;
     case DBG_STEP:
      fDebug(HIGH,("fDebugMonitor(): returned STEP"));
      bGblPauseMode=T_FALSE;
      bStep=T_TRUE;
      break;
     case DBG_STEPRUN:
      fDebug(HIGH,("fDebugMonitor(): returned STEPRUN"));
      bGblPauseMode=T_FALSE;
      bStepRun=T_TRUE;
      break;
     default:
      fDebug(HIGH,("fDebugMonitor(): returned unknown value"));
      break;
    }
    break;
   } //if (strcmp(apcArgV[0],asCommandList[iTmp].pcCommandName)==0)
   /* Check for the end of command array and print command not */
   /* found error and exit from this command scan loop         */

   iTmp++;
   if (asCommandList[iTmp].pfCommandFunc==NULL)
   {
    edit_log_add(get_window(MonitorWinId),"Command not found.");
    edit_log_add(get_window(MonitorWinId),"");
    break;
   }
  }
  aktivate_window(get_window(MonitorWinId));
 }
 return EDIT_CONTINUE;
}

T_UCHAR frecal_check(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[])
{
 rc_cycles=0;
 recal_check=INIT_RECAL_CHECK;
 pause_on_monitor_entry=T_FALSE;
 bStepRun=T_FALSE;
 bStep=T_FALSE;
 bGblPauseMode=T_FALSE;
 return(DBG_OK);
}

T_UCHAR fcycles(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[])
{
 static T_ULONG cycles =0;

 if (iArgC>2)
 {
  edit_log_add(get_window(MonitorWinId),"Invalid argument.\nUsage: CYCLES [value]");
  edit_log_add(get_window(MonitorWinId),"");
  return(DBG_ERROR);
 }
 if (iArgC==2)
 {
  if (fStringToLongWord(apcArgV[1],&cycles)==T_FALSE)
  {
   edit_log_add(get_window(MonitorWinId),"Invalid value.");
   edit_log_add(get_window(MonitorWinId),"");
   return(DBG_ERROR);
  }
  cycles=-cycles+lwGblSystemTickTotal;
  sprintf(string_buffer,"CYCLES set to:%i",lwGblSystemTickTotal-cycles);
  edit_log_add(get_window(MonitorWinId),string_buffer);
  edit_log_add(get_window(MonitorWinId),"");
 }
 else
 {
  sprintf(string_buffer,"CYCLES now at:%i",lwGblSystemTickTotal-cycles);
  edit_log_add(get_window(MonitorWinId),string_buffer);
  edit_log_add(get_window(MonitorWinId),"");
 }
 return(DBG_OK);
}

T_INT watch_destructor(PARAMETER *parameter)
{
 WINDOW_LISTEN_ELEMENT *window=(WINDOW_LISTEN_ELEMENT *)parameter->window_element;
 if (parameter==NULL)
 {
  show_message("No Parameter in DESTRUCTOR()");
  return -1;
 }
 if (window==NULL)
 {
  show_message("I can't find myself!!! (Window not closed)");
  return -1;
 }
 // interesting part of destructor
 if (window->user_data!=NULL)
 {
  d_entferne_aus_liste(window->user_data,&wliste_first,&wliste_last,DL_CLEAN_UP);
  free(window->user_data);
 }
 if (wliste_first==NULL)
 {
  remove_update_funktion(watch_update_function);
 }
 window->destructor=NULL; // the rest is done by close_window
 window->parameter=NULL;
 return -1; // keine variablen da, dann ebene = -1
}

T_UCHAR fwatch(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[])
{
 OPEN_WIN w;
 WINDOW_LISTEN_ELEMENT *window;
 VARIABLE_WATCH *watch;
 T_LISTE *liste;
 T_UWORD address;
 T_PCHAR label_name=NULL;
 T_CHAR bin[129];
 T_ULONG h;
 T_INT old_print_modus;
 T_INT old_text;
 T_INT old_back_color;
 T_INT old_text_color;
 old_print_modus=vesa_print_modus;
 old_text=vesa_text;
 old_back_color=vesa_back_color;
 old_text_color=vesa_text_color;
 if (iArgC!=2)
 {
  edit_log_add(get_window(MonitorWinId),"Invalid argument.\nUsage: WATCH string|address");
  edit_log_add(get_window(MonitorWinId),"");
  return(DBG_ERROR);
 }
 if (dis_init_state==T_FALSE)
 {
  dis_init_state=T_TRUE;
  init_dis();
 }
 // now first look if it is a label of some kind...
 liste=info_liste_first;
 while (liste!=NULL)
 {
  D_ADDRESS_INFO *info;
  info=(D_ADDRESS_INFO *)(liste->objekt);
  if (info->kind_of_info==D_INFO_LABEL)
  {
   T_INT cmp_len;
   if (strlen(info->string)>strlen(apcArgV1[1]))
    cmp_len=strlen(info->string);
   else
    cmp_len=strlen(apcArgV1[1]);
   if (strncmp(info->string,apcArgV1[1],cmp_len)==0)
   {
    address=info->address;
    label_name=info->string;
    break;
   }
  }
  liste=liste->naechstes;
 }
 if (liste==NULL)
 {
  if (fStringToWord(apcArgV[1],&address)==T_FALSE)
  {
   edit_log_add(get_window(MonitorWinId),"Invalid watch value.");
   edit_log_add(get_window(MonitorWinId),"");
   return(DBG_ERROR);
  }
 }
 watch=(VARIABLE_WATCH *)malloc(sizeof(VARIABLE_WATCH));
 if (watch==NULL)
 {
  fFatal("fwatch() out of memory");
 }
 watch->len=T_BYTE_LEN;
 watch->name=label_name;
 watch->pointer=&(atSystemImage[address]);
 watch->old_value=*((T_PUCHAR)(watch->pointer));

 w.x_pos=0;
 w.y_pos=0;
 w.breite=14*8+4;
 w.hoehe=4*16+2;
 w.border=2;
 w.color=ddark_grau;
 w.bcolor=dhell_grau;
 w.art=RAW_WINDOW|WITH_CLOSE_BUTTON|WITH_HIDE_BUTTON|REMEMBER;
 w.id=WIN_OPEN_CHOOSES_ID;
 w.variablen=NULL;
 w.mem_bild=NULL;
 w.pcx_bild=NULL;
 w.name=label_name;
 w.destructor=watch_destructor;
 w.area_funktion=execute_win_move;
 w.tasten_funktion=NULL;
 MOUSE_VERSTECKE
 window=open_window(&w);
 if (window==NULL)
 {
  show_message("Watch Window opening error!");
  MOUSE_ZEIGE
  return T_FALSE;
 }
 print_srect(window,PRINT_WIN_RAW,0,0,window->b,window->h,ddark_grau,weiss,2);
 h=*((T_PUCHAR)(watch->pointer));
 vesa_print_modus=PIC_FILLED;
 vesa_text=SINGLE_COLOR;
 vesa_back_color=dhell_grau;
 vesa_text_color=ddark_grau;
 wprinta1(window,2,  2+0*16, "$%02x",h);
 wprinta1(window,2,  2+1*16, "#%03i",h);
 wprinta1(window,2,  2+2*16, "%%%s    ",itob(h,bin));
 if (h!=0)
  wprinta1(window,2,  2+3*16, "'%c' ",(T_UCHAR)h);
 else
  wprinta1(window,2,  2+3*16, "'' ",(T_UCHAR)h);
 MOUSE_ZEIGE
 watch->window_id=window->id;
 window->user_data=(T_PVOID)watch;
 if (wliste_first==NULL)
 {
  if (get_window(MonitorWinId))
  {
   remove_update_funktion(monitor_update_function);
   insert_update_funktion(watch_update_function,1);
   monitor_update=insert_update_funktion(monitor_update_function,1);
  }
  else
   insert_update_funktion(watch_update_function,1);
 }
 d_fuege_in_liste_ein( watch, &wliste_first, &wliste_last, DL_HINTEN );
 vesa_print_modus=old_print_modus;
 vesa_text=old_text;
 vesa_back_color=old_back_color;
 vesa_text_color=old_text_color;
 return(DBG_OK);
}

T_VOID watch_update_function(PARAMETER *parameter)
{
 WINDOW_LISTEN_ELEMENT *window;
 VARIABLE_WATCH *watch;
 T_CHAR bin[129];
 T_ULONG h;
 T_LISTE *liste=wliste_first;
 T_INT old_print_modus;
 T_INT old_text;
 T_INT old_back_color;
 T_INT old_text_color;
 old_print_modus=vesa_print_modus;
 old_text=vesa_text;
 old_back_color=vesa_back_color;
 old_text_color=vesa_text_color;
 vesa_print_modus=PIC_FILLED;
 vesa_text=SINGLE_COLOR;
 vesa_back_color=dhell_grau;
 vesa_text_color=ddark_grau;
 while (liste!=NULL)
 {
  watch=(VARIABLE_WATCH *)liste->objekt;
  if (watch!=NULL)
  {
   window=get_window(watch->window_id);
   if (watch->old_value!=*((T_PUCHAR)(watch->pointer)))
   {
    h=*((T_PUCHAR)(watch->pointer));
    wprinta1(window,2,  2+0*16, "$%02x",h);
    wprinta1(window,2,  2+1*16, "#%03i",h);
    wprinta1(window,2,  2+2*16, "%%%s    ",itob(h,bin));
    if (h!=0)
     wprinta1(window,2,  2+3*16, "'%c' ",(T_UCHAR)h);
    else
     wprinta1(window,2,  2+3*16, "'' ",(T_UCHAR)h);
    watch->old_value=(T_UCHAR)h;
   }
  } // if (watch!=NULL)
  liste=liste->naechstes;
 } // while (wliste_first!=NULL)
 vesa_print_modus=old_print_modus;
 vesa_text=old_text;
 vesa_back_color=old_back_color;
 vesa_text_color=old_text_color;
}

T_UCHAR fpoke(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[])
{
 T_UWORD address;
 T_PCHAR label_name=NULL;
 T_UWORD value;
 T_LISTE *liste;
 T_INT bTmp;
 if (iArgC!=3)
 {
  edit_log_add(get_window(MonitorWinId),"Invalid argument.\nUsage: POKE string|address, value");
  edit_log_add(get_window(MonitorWinId),"");
  return(DBG_ERROR);
 }
 if (dis_init_state==T_FALSE)
 {
  dis_init_state=T_TRUE;
  init_dis();
 }
 // now first look if it is a label of some kind...
 liste=info_liste_first;
 while (liste!=NULL)
 {
  D_ADDRESS_INFO *info;
  info=(D_ADDRESS_INFO *)(liste->objekt);
  if (info->kind_of_info==D_INFO_LABEL)
  {
   T_INT cmp_len;
   if (strlen(info->string)>strlen(apcArgV1[1]))
    cmp_len=strlen(info->string);
   else
    cmp_len=strlen(apcArgV1[1]);
   if (strncmp(info->string,apcArgV1[1],cmp_len)==0)
   {
    address=info->address;
    label_name=info->string;
    break;
   }
  }
  liste=liste->naechstes;
 }
 if (liste==NULL)
 {
  if (fStringToWord(apcArgV[1],&address)==T_FALSE)
  {
   edit_log_add(get_window(MonitorWinId),"Invalid poke address.");
   edit_log_add(get_window(MonitorWinId),"");
   return(DBG_ERROR);
  }
 }
 if (fStringToWord(apcArgV[2],&value)==T_FALSE)
 {
  edit_log_add(get_window(MonitorWinId),"Invalid poke value.");
  edit_log_add(get_window(MonitorWinId),"");
  return(DBG_ERROR);
 }
 bTmp=bGblCartridgeROMWriteEnable;
 bGblCartridgeROMWriteEnable=T_TRUE;
 f6809AddressWrite(address,value);
 bGblCartridgeROMWriteEnable=bTmp;
 sprintf(string_buffer,"POKED %i to:%i",value, address);
 edit_log_add(get_window(MonitorWinId),string_buffer);
 edit_log_add(get_window(MonitorWinId),"");
 return(DBG_OK);
}

T_UCHAR flabels(T_SWORD iArgC,T_PUCHAR apcArgV[],T_PUCHAR apcArgV1[])
{
 T_LISTE *liste;
 T_INT i=0;
 if (dis_init_state==T_FALSE)
 {
  dis_init_state=T_TRUE;
  init_dis();
 }
 // now first look if it is a label of some kind...
 edit_log_add(get_window(MonitorWinId),"");
 edit_log_add(get_window(MonitorWinId),"Control file labels:");
 liste=info_liste_first;
 while (liste!=NULL)
 {
  D_ADDRESS_INFO *info;
  info=(D_ADDRESS_INFO *)(liste->objekt);
  if (info->kind_of_info==D_INFO_LABEL)
  {
   sprintf(string_buffer,"Label: \"%s\", Value $%04X",info->string, info->address);
   edit_log_add(get_window(MonitorWinId),string_buffer);
  }
  liste=liste->naechstes;
 }
 edit_log_add(get_window(MonitorWinId),"");
 edit_log_add(get_window(MonitorWinId),"Internal Chip labels:");
 while (vlist[i].name!=NULL)
 {
  edit_log_add(get_window(MonitorWinId),vlist[i].name);
  i++;
 } // while (vlist[i].name!=NULL)
 return(DBG_OK);
}



