/*****************************************************************************/
/*                                                                           */
/*   VECTREX HP3000 Software Emulator              Copyright K.Wilkins 1996  */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*   Title      : Vectrex emulator sequencing module functions               */
/*                                                                           */
/*   File Name  : SEQUENCE.C                                                 */
/*                                                                           */
/*   Author     : Keith Wilkins / C.S.                                       */
/*                                                                           */
/*   Version    : 1.20                                                       */
/*                                                                           */
/*   Desciption : This file contains the functions for the Vectrex sequencer */
/*                module. This module executes all of the emulator sub-func- */
/*                tions at the correct times depending on the various tick   */
/*                counters that are used to regulate relative timing.        */
/*                                                                           */
/*   Functions  : fVectrexEmulate() - Run the emulator for a while           */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*   Revision History:                                                       */
/*                                                                           */
/*   Version    Date    Who  Description of changes                          */
/*   -------    ----    ---  ----------------------                          */
/*                                                                           */
/*    0.01    14/03/93  K.W  Creation of empty file and descriptions         */
/*                                                                           */
/*    0.02    17/03/93  K.W  1st Stab at the sequencer                       */
/*                                                                           */
/*    1.00    05/08/96  K.W  Public release of DVE V1.0                      */
/*                                                                           */
/*    1.20    20/10/96  C.S  MultiTick and inline stuff                      */
/*                                                                           */
/*****************************************************************************/
#ifdef _NO_DEBUG_INFORMATION_

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <standard.h>
#include <tools.h>

#include "keyboard.h"
#include "sound.h"
#include "emu6809.h"
#include "emu6522.h"
#include "analog.h"
#include "sys_ram.h"
#include "sys_rom.h"
#include "cart_rom.h"
#include "display.h"
#include "verror.h"
#include "vdebug.h"
#include "monitor.h"
#include "vectrex.h"
#include "io.h"
#include "sequence.h"

T_SWORD tTicker=10;

T_INT (* fVectrexEmulate)(T_VOID)=NULL;

T_ULONG counter_since_change=0;
T_INT state=1;
T_INT wheel_color_rotator=0;
T_INT frstate=0;

#ifdef CLAY_ONLY
T_INT cscreenshots=T_FALSE;
T_INT cgrabsdone=0;
#endif

T_INT update_debug_window(PARAMETER *parameter);

T_VOID fopen_monitor_debug_pause(T_VOID)
{
 if (bGblDebugEnable==T_FALSE)
 {
  fDebugToggle(0);
 }
 if (bGblDebugMonitorEnable==T_FALSE)
 {
  fDebugMonitorToggle(0);
 }
 if (bGblPauseMode==T_FALSE)
 {
  bGblPauseMode=T_TRUE;
  vectrex_update->frequency=1;
  vector_undraw();
  vector_refresh();
 }
}
T_VOID firstrun(T_VOID)
{
 if (frstate==0)
 {
  frstate=1;
  /************************/
  /* Install control keys */
  /************************/
  fKBDfuncreg(KEY_SPACE,fGUIToggle);

  fKBDfuncreg(KEY_ALT_M,fDebugMonitorToggle);
  fKBDfuncreg(KEY_ALT_D,fDebugToggle);
  fKBDfuncreg(KEY_ALT_F,fDebugFileToggle);
  fKBDfuncreg(KEY_SHIFT_F1,fDebugLevelLow);
  fKBDfuncreg(KEY_SHIFT_F2,fDebugLevelMid);
  fKBDfuncreg(KEY_SHIFT_F3,fDebugLevelHigh);

  fKBDfuncreg(KEY_ALT_X,fSequencerAbort);
  fKBDfuncreg(KEY_ALT_R,fSequencerReset);
  fKBDfuncreg(KEY_ALT_B,fSequencerReboot);
  fKBDfuncreg(KEY_ALT_P,fSequencerPause);
  fKBDfuncreg(KEY_ALT_G,fScreenshot);
  fKBDfuncreg(KEY_ALT_E,fEnableEvent);

  fKBDfuncreg(KEY_0,fLoadState);
  fKBDfuncreg(KEY_1,fLoadState);
  fKBDfuncreg(KEY_2,fLoadState);
  fKBDfuncreg(KEY_3,fLoadState);
  fKBDfuncreg(KEY_4,fLoadState);
  fKBDfuncreg(KEY_5,fLoadState);
  fKBDfuncreg(KEY_6,fLoadState);
  fKBDfuncreg(KEY_7,fLoadState);
  fKBDfuncreg(KEY_8,fLoadState);
  fKBDfuncreg(KEY_9,fLoadState);

  fKBDfuncreg(KEY_ALT_1,fSaveState);
  fKBDfuncreg(KEY_ALT_2,fSaveState);
  fKBDfuncreg(KEY_ALT_3,fSaveState);
  fKBDfuncreg(KEY_ALT_4,fSaveState);
  fKBDfuncreg(KEY_ALT_5,fSaveState);
  fKBDfuncreg(KEY_ALT_6,fSaveState);
  if (keymap==USA_KEYMAP)
  {
   fKBDfuncreg(KEY_ALT_0,fSaveState);
   fKBDfuncreg(KEY_ALT_7,fSaveState);
   fKBDfuncreg(KEY_ALT_8,fSaveState);
   fKBDfuncreg(KEY_ALT_9,fSaveState);
  }
  else if (keymap==GERMAN_KEYMAP)
  {
   fKBDfuncreg(KEY_RCURLY,fSaveState);
   fKBDfuncreg(KEY_LCURLY,fSaveState);
   fKBDfuncreg(KEY_LSQRB,fSaveState);
   fKBDfuncreg(KEY_RSQRB,fSaveState);
  }
#ifdef CLAY_ONLY
  fKBDfuncreg(KEY_ALT_C,fCScreenshot);
#endif
  fKBDfuncreg(KEY_ALT_F1 ,fCarouselSelect);
  fKBDfuncreg(KEY_ALT_F2 ,fCarouselSelect);
  fKBDfuncreg(KEY_ALT_F3 ,fCarouselSelect);
  fKBDfuncreg(KEY_ALT_F4 ,fCarouselSelect);
  fKBDfuncreg(KEY_ALT_F5 ,fCarouselSelect);
  fKBDfuncreg(KEY_ALT_F6 ,fCarouselSelect);
  fKBDfuncreg(KEY_ALT_F7 ,fCarouselSelect);
  fKBDfuncreg(KEY_ALT_F8 ,fCarouselSelect);
  fKBDfuncreg(KEY_ALT_F9 ,fCarouselSelect);
  fKBDfuncreg(KEY_ALT_F10,fCarouselSelect);
  fKBDfuncreg(KEY_ALT_F11,fCarouselSelect);
  fKBDfuncreg(KEY_ALT_F12,fCarouselSelect);
 }
 if (reset_state==T_TRUE)
 {
  f6809Reset();
 }
 if (reset_state==T_TRUE)
 {
  if (bGblRestoreOnStart==T_TRUE)
  {
   fLoadState(0);
  }
 }
 reset_state=T_TRUE;
 tTicker=10;
}

/*****************************************************************************/
/*                                                                           */
/*   Function    : fPauseSequencer()                                         */
/*                                                                           */
/*   Author      : K.W                                                       */
/*                                                                           */
/*   Description : Halt the sequencer temporarily                            */
/*                                                                           */
/*   Accepts     : WORD key index that caused the call                       */
/*                                                                           */
/*   Returns     : nothing                                                   */
/*                                                                           */
/*****************************************************************************/
T_VOID fSequencerPause(T_UWORD tKeyPress)
{
 tKeyPress=0;
/*
 bGblStatsValid=T_FALSE;
 fSoundMute();
 fKBDbuffering(T_TRUE);
 fKBDgetch();
 fKBDbuffering(T_FALSE);
 fSoundResume();
*/
 if (bGblGUIEnable==T_FALSE)
 {
  fGUIToggle(0);
 }
 bGblPauseMode=T_TRUE;
 vector_undraw();
 vector_refresh();

 vectrex_update->frequency=1;
}

/*****************************************************************************/
/*                                                                           */
/*   Function    : fAbortSequencer()                                         */
/*                                                                           */
/*   Author      : K.W                                                       */
/*                                                                           */
/*   Description : Set the sequencer abort flag to force the system to exit. */
/*                                                                           */
/*   Accepts     : WORD key index that caused the call                       */
/*                                                                           */
/*   Returns     : nothing                                                   */
/*                                                                           */
/*****************************************************************************/
T_VOID fSequencerAbort(T_UWORD tKeyPress)
{
 tKeyPress=0;
 wGblQuitFlag=EMU_DONE;
}

T_VOID fScreenshot(T_UWORD tKeyPress)
{
 FILE *this_file=NULL;
#ifndef CLAY_ONLY
 T_CHAR *filename="screen00.pcx";
#endif
 T_CHAR fname[8+3+1+1];
#ifdef CLAY_ONLY
 sprintf(fname,"%08i.pcx",cgrabsdone);
#else
 strcpy(fname,filename);
 while ((this_file = fopen(fname, "rb"))!= NULL)
 {
  fclose(this_file);
  fname[7]++;
  if (fname[7]=='9'+1)
  {
   fname[7]='0';
   fname[6]++;
   if (fname[6]=='9'+1)
   {
    fFatal("seq0 Grab PCX Screen() - Filenames all used!");
   }
  }
 }
#endif
 tKeyPress=0;
 if (bGblGUIEnable==T_TRUE)
 {
  MOUSE_VERSTECKE
  hide_all_windows();
 }
 pcx_write_256pic(0,0,vesa_x,vesa_y,fname);
 if (bGblGUIEnable==T_TRUE)
 {
  show_all_windows();
  MOUSE_ZEIGE
 }
}

#ifdef CLAY_ONLY
T_VOID fCScreenshot(T_UWORD tKeyPress)
{
 tKeyPress=0;
 if (cscreenshots==T_TRUE)
 {
  cscreenshots=T_FALSE;
  fIOPrintf(0,0,0,0,1,"      ");
  fIOPrintf(0,20,0,0,1,"        ");
 }
 else
  cscreenshots=T_TRUE;
}
#endif

#endif //#ifdef _NO_DEBUG_INFORMATION_

#ifndef _NO_DEBUG_INFORMATION_

T_VOID fEnableEvent(T_UWORD tKeyPress)
{
 tKeyPress=0;
 enable_event=(enable_event+1)%2;
 if (enable_event==1)
  bGblForceSingleProcessor=T_TRUE;
 else
  bGblForceSingleProcessor=T_FALSE;
}

T_VOID fSaveState(T_UWORD tKeyPress)
{
 FILE *saveram=NULL;
 T_UINT i;
 T_CHAR filename[80];
 T_PCHAR current_path=NULL;

 if (strcmp(acGblDefaultCartFileName,""))
  strcpy(filename,acGblDefaultCartFileName);
 else
  strcpy(filename,acGblROMImageFileName);
 if (strrchr(filename,'.')!=NULL)
  *(strrchr(filename,'.'))='\0';

 if ((tKeyPress>=KEY_ALT_1)&&(tKeyPress<=KEY_ALT_9))
 {
  strcat(filename,".sv ");
  filename[strlen(filename)-1]='1'+tKeyPress-KEY_ALT_1;
 }
 else
 {
  if ((keymap==GERMAN_KEYMAP)&&((tKeyPress==KEY_LCURLY)||(tKeyPress==KEY_LSQRB)||(tKeyPress==KEY_RSQRB)||(tKeyPress==KEY_RCURLY)))
  {
   if (tKeyPress==KEY_RCURLY)
    strcat(filename,".sav");
   if (tKeyPress==KEY_LCURLY)
    strcat(filename,".sv7");
   if (tKeyPress==KEY_LSQRB)
    strcat(filename,".sv8");
   if (tKeyPress==KEY_RSQRB)
    strcat(filename,".sv9");
  }
  else
   strcat(filename,".sav");
 }
 current_path=get_current_path();
 if (GblSvgDir!=NULL)
 {
//  if (f_file_exist(GblSvgDir)==T_TRUE)
   change_directory(GblSvgDir);
 }
 saveram = fopen(filename, "w+b"); /* file wird ueberschrieben */
 if (saveram==NULL)
 {
  if (current_path!=NULL)
  {
   change_directory(current_path);
   free(current_path);
   current_path=NULL;
  }
  return;
 }
 /***/
 for (i=12*4096+8*256;i<13*4096;i++)
  fwrite((T_CHAR *)(atSystemImage+i),1,1,saveram);

 if (bGblExtendedRam==T_TRUE)          /* Animaction Ram 2K */
 {
  for (i=2*4096;i<2*4096+8*256;i++)
   fwrite((T_CHAR *)(atSystemImage+i),1,1,saveram);
 }
 /***/

 /***/
 fwrite(&lXSampleHold,1,sizeof(lXSampleHold),saveram);
 fwrite(&lYSampleHold,1,sizeof(lYSampleHold),saveram);
 fwrite(&tZSampleHold,1,sizeof(tZSampleHold),saveram);

 fwrite(&lSIGDEF_XOUTPUT,1,sizeof(lSIGDEF_XOUTPUT),saveram);
 fwrite(&lSIGDEF_YOUTPUT,1,sizeof(lSIGDEF_YOUTPUT),saveram);
 fwrite(&tSIGDEF_ZOUTPUT,1,sizeof(tSIGDEF_ZOUTPUT),saveram);
 /***/

 /***/
 fwrite(&t6522ORB_0,1,sizeof(t6522ORB_0),saveram);
 fwrite(&t6522IRB_0,1,sizeof(t6522IRB_0),saveram);

 fwrite(&t6522IRB_0L,1,sizeof( t6522IRB_0L),saveram);

 fwrite(&t6522ORB_1,1,sizeof(t6522ORB_1),saveram);
 fwrite(&t6522IRB_1,1,sizeof(t6522IRB_1),saveram);
 fwrite(&t6522IRB_1L,1,sizeof(t6522IRB_1L),saveram);

 fwrite(&t6522ORB_2,1,sizeof(t6522ORB_2),saveram);
 fwrite(&t6522IRB_2,1,sizeof(t6522IRB_2),saveram);
 fwrite(&t6522IRB_2L,1,sizeof(t6522IRB_2L),saveram);

 fwrite(&t6522ORB_3,1,sizeof(t6522ORB_3),saveram);
 fwrite(&t6522IRB_3,1,sizeof(t6522IRB_3),saveram);
 fwrite(&t6522IRB_3L,1,sizeof(t6522IRB_3L),saveram);

 fwrite(&t6522ORB_4,1,sizeof(t6522ORB_4),saveram);
 fwrite(&t6522IRB_4,1,sizeof(t6522IRB_4),saveram);
 fwrite(&t6522IRB_4L,1,sizeof(t6522IRB_4L),saveram);

 fwrite(&t6522ORB_5,1,sizeof(t6522ORB_5),saveram);
 fwrite(&t6522IRB_5,1,sizeof(t6522IRB_5),saveram);
 fwrite(&t6522IRB_5L,1,sizeof(t6522IRB_5L),saveram);

 fwrite(&t6522ORB_6,1,sizeof(t6522ORB_6),saveram);
 fwrite(&t6522IRB_6,1,sizeof(t6522IRB_6),saveram);
 fwrite(&t6522IRB_6L,1,sizeof(t6522IRB_6L),saveram);

 fwrite(&t6522ORB_7REG,1,sizeof(t6522ORB_7REG),saveram);
 fwrite(&t6522ORB_7,1,sizeof(t6522ORB_7),saveram);
 fwrite(&t6522IRB_7,1,sizeof(t6522IRB_7),saveram);
 fwrite(&t6522IRB_7L,1,sizeof(t6522IRB_7L),saveram);

 fwrite(&t6522ORA,1,sizeof(t6522ORA),saveram);
 fwrite(&t6522IRA,1,sizeof(t6522IRA),saveram);
 fwrite(&t6522IRAL,1,sizeof(t6522IRAL),saveram);

 fwrite(&t6522DDRB,1,sizeof(t6522DDRB),saveram);
 fwrite(&t6522DDRA,1,sizeof(t6522DDRA),saveram);

 fwrite(&t6522SHFTREG,1,sizeof(t6522SHFTREG),saveram);
 fwrite(&t6522SHFTREGACT,1,sizeof(t6522SHFTREGACT),saveram);

 fwrite(&u6522Timer1,1,sizeof(TIMER),saveram);
 fwrite(&u6522Timer2,1,sizeof(TIMER),saveram);
 fwrite(&bTimer2OneShot,1,sizeof(T_INT),saveram);

 fwrite(&t6522AUX_T1CTRL,1,sizeof(t6522AUX_T1CTRL),saveram);
 fwrite(&t6522AUX_T2CTRL,1,sizeof(t6522AUX_T2CTRL),saveram);
 fwrite(&t6522AUX_SHFTCTRL,1,sizeof(t6522AUX_SHFTCTRL),saveram);
 fwrite(&t6522AUX_PA_LE,1,sizeof(t6522AUX_PA_LE),saveram);
 fwrite(&t6522AUX_PB_LE,1,sizeof(t6522AUX_PB_LE),saveram);

 fwrite(&t6522PCR_CB2CTRL,1,sizeof(t6522PCR_CB2CTRL),saveram);
 fwrite(&t6522PCR_CB1CTRL,1,sizeof(t6522PCR_CB1CTRL),saveram);
 fwrite(&t6522PCR_CA2CTRL,1,sizeof(t6522PCR_CA2CTRL),saveram);
 fwrite(&t6522PCR_CA1CTRL,1,sizeof(t6522PCR_CA1CTRL),saveram);

 fwrite(&t6522IER,1,sizeof(t6522IER),saveram);

 fwrite(&t6522IFLG,1,sizeof(t6522IFLG),saveram);

 fwrite(&t6522CA1,1,sizeof(t6522CA1),saveram);
 fwrite(&t6522CA2,1,sizeof(t6522CA2),saveram);
 fwrite(&t6522CB1,1,sizeof(t6522CB1),saveram);
 fwrite(&t6522CB2,1,sizeof(t6522CB2),saveram);

 fwrite(&tsCPURegisters,1,sizeof(tsCPURegisters),saveram);
 fwrite(&iToggle,1,sizeof(iToggle),saveram);
 fwrite(&tcount,1,sizeof(tcount),saveram);
 fwrite(&t1int,1,sizeof(t1int),saveram);
 /***/

 /***/
 fwrite(&iInstruction,1,sizeof(iInstruction),saveram);

 fwrite(&i6809status,1,sizeof(i6809status),saveram);
 fwrite(&iLastAddress,1,sizeof(iLastAddress),saveram);
 /***/

 /***/
// fwrite(&lwGblSystemTickTotal,1,sizeof(lwGblSystemTickTotal),saveram);
// fwrite(&lwGblDisplayTickCounter,1,sizeof(lwGblDisplayTickCounter),saveram);
// fwrite(&tick_timeout,1,sizeof(tick_timeout),saveram);
// fwrite(&last_timer_tick,1,sizeof(last_timer_tick),saveram);
 /***/

 /***/
 fwrite(&tOLD_BC1,1,sizeof(tOLD_BC1),saveram);
 fwrite(&tOLD_BDIR,1,sizeof(tOLD_BDIR),saveram);
 fwrite(&tAddr,1,sizeof(tAddr),saveram);
 /***/

 /***/
// /* chris
 fwrite(&iLineSelected,1,sizeof(iLineSelected),saveram);
 fwrite(&iLineSelector,1,sizeof(iLineSelector),saveram);
 fwrite(&lwRefreshMarker,1,sizeof(lwRefreshMarker),saveram);

 for (i=0;i<LINESTORE_DEPTH;i++)
  fwrite(&(line_store[0])+i*(sizeof(SDLINE)),1,sizeof(SDLINE),saveram);

 fwrite(&line_freelist,1,sizeof(line_freelist),saveram);
 fwrite(&line_rootnode,1,sizeof(line_rootnode),saveram);

 fwrite(hash_rootnodes,HASH_TABLE_SIZE,sizeof(T_SWORD),saveram);
 fwrite(&hash_freelist,1,sizeof(T_SWORD),saveram);

 for (i=0;i<LINESTORE_DEPTH;i++)
  fwrite(&(hash_store[0])+i*(sizeof(SHASH)),1,sizeof(SHASH),saveram);
// */
 /***/
 fclose(saveram);
 if (current_path!=NULL)
 {
  change_directory(current_path);
  free(current_path);
  current_path=NULL;
 }
 tKeyPress=0;
}

T_VOID fLoadState(T_UWORD tKeyPress)
{
// not debug
 FILE *loadram=NULL;
 T_UINT i;
 T_SWORD loop;
 PIXEL screen;
 T_PCHAR current_path=NULL;

 T_CHAR filename[80];

 if(strcmp(acGblDefaultCartFileName,""))
  strcpy(filename,acGblDefaultCartFileName);
 else
  strcpy(filename,acGblROMImageFileName);
 if(strrchr(filename,'.')!=NULL)
  *(strrchr(filename,'.'))='\0';

 if ((tKeyPress>=KEY_1)&&(tKeyPress<=KEY_9))
 {
  strcat(filename,".sv ");
  filename[strlen(filename)-1]='1'+tKeyPress-KEY_1;
 }
 else
 {
  strcat(filename,".sav");
 }
 current_path=get_current_path();
 if (GblSvgDir!=NULL)
  change_directory(GblSvgDir);
 if (f_file_exist(filename)==T_FALSE)
 {
  if (current_path!=NULL)
  {
   change_directory(current_path);
   free(current_path);
   current_path=NULL;
  }
 }
 loadram = fopen(filename, "rb");
 if (loadram==NULL)
 {
  if (current_path!=NULL)
  {
   change_directory(current_path);
   free(current_path);
   current_path=NULL;
  }
  return;
 }
 if (current_path!=NULL)
 {
  change_directory(current_path);
 }
 if ((bGblGUIEnable==T_TRUE)||(bGblLightpenEnable==T_TRUE))
 {
  MOUSE_VERSTECKE
 }
 bGblDisplayFullRefresh=T_FALSE;
 bGblLightpenEnable=T_FALSE;
 bGblGoggleGame=T_FALSE;
 rectengle_clipping_enable=T_TRUE;
 ImagerColorNow=255;
 wGblImagerMode=0;
 wGblColor1Degrees=60;
 wGblColor2Degrees=60;
 wGblColor3Degrees=60;
 wGblImagerXOffset=0;
 wGblImagerXOffset1=0;
 wGblImagerXOffset2=0;

 lwGblClipX=lwGblClipX_reset;               // Clipping window top right
 lwGblClipY=lwGblClipY_reset;

 divider_x[0]=divider_x_reset[0];
 divider_y[0]=divider_y_reset[0];

 divider_x[1]=divider_x_reset[1];
 divider_y[1]=divider_y_reset[1];

 divider_x[2]=divider_x_reset[2];
 divider_y[2]=divider_y_reset[2];

 divider_x[3]=divider_x_reset[3];
 divider_y[3]=divider_y_reset[3];

 divider_x[4]=divider_x_reset[4];
 divider_y[4]=divider_y_reset[4];

 divider_x[5]=divider_x_reset[5];
 divider_y[5]=divider_y_reset[5];

 display_infox[0].iDivShiftA=display_infox_reset[0].iDivShiftA;
 display_infox[0].iDivShiftB=display_infox_reset[0].iDivShiftB;
 display_infoy[0].iDivShiftA=display_infoy_reset[0].iDivShiftA;
 display_infoy[0].iDivShiftB=display_infoy_reset[0].iDivShiftB;

 display_infox[1].iDivShiftA=display_infox_reset[1].iDivShiftA;
 display_infox[1].iDivShiftB=display_infox_reset[1].iDivShiftB;
 display_infoy[1].iDivShiftA=display_infoy_reset[1].iDivShiftA;
 display_infoy[1].iDivShiftB=display_infoy_reset[1].iDivShiftB;

 display_infox[2].iDivShiftA=display_infox_reset[2].iDivShiftA;
 display_infox[2].iDivShiftB=display_infox_reset[2].iDivShiftB;
 display_infoy[2].iDivShiftA=display_infoy_reset[2].iDivShiftA;
 display_infoy[2].iDivShiftB=display_infoy_reset[2].iDivShiftB;

 display_infox[3].iDivShiftA=display_infox_reset[3].iDivShiftA;
 display_infox[3].iDivShiftB=display_infox_reset[3].iDivShiftB;
 display_infoy[3].iDivShiftA=display_infoy_reset[3].iDivShiftA;
 display_infoy[3].iDivShiftB=display_infoy_reset[3].iDivShiftB;

 display_infox[4].iDivShiftA=display_infox_reset[4].iDivShiftA;
 display_infox[4].iDivShiftB=display_infox_reset[4].iDivShiftB;
 display_infoy[4].iDivShiftA=display_infoy_reset[4].iDivShiftA;
 display_infoy[4].iDivShiftB=display_infoy_reset[4].iDivShiftB;

 display_infox[5].iDivShiftA=display_infox_reset[5].iDivShiftA;
 display_infox[5].iDivShiftB=display_infox_reset[5].iDivShiftB;
 display_infoy[5].iDivShiftA=display_infoy_reset[5].iDivShiftA;
 display_infoy[5].iDivShiftB=display_infoy_reset[5].iDivShiftB;

 screen=fIOGetScreenRes();
 iXOffset=screen.iXpos/2;
 iYOffset=screen.iYpos/2;
 if (bGblDisplayExactScaling==T_FALSE)
 {
  iX1clip=(lwGblClipX>>iDivShiftAx) + (lwGblClipX>>iDivShiftBx);
  iY1clip=(lwGblClipY>>iDivShiftAy) + (lwGblClipY>>iDivShiftBy);
  iX2clip=(-lwGblClipX>>iDivShiftAx) + (-lwGblClipX>>iDivShiftBx);
  iY2clip=(-lwGblClipY>>iDivShiftAy) + (-lwGblClipY>>iDivShiftBy);

  mausxmin=iXOffset-((20000>>iDivShiftAx)+(20000>>iDivShiftBx));
  mausxmax=iXOffset+((20000>>iDivShiftAx)+(20000>>iDivShiftBx));
  mausymin=iYOffset-((20000>>iDivShiftAy)+(20000>>iDivShiftBy));
  mausymax=iYOffset+((20000>>iDivShiftAy)+(20000>>iDivShiftBy));
 }
 else
 {
  iX1clip=lwGblClipX/dividerx;
  iY1clip=lwGblClipY/dividery;
  iX2clip=-lwGblClipX/dividerx;
  iY2clip=-lwGblClipY/dividery;

  mausxmin=iXOffset-20000/dividerx; // just in case a lightpen
  mausxmax=iXOffset+20000/dividerx; // will be enabled
  mausymin=iYOffset-20000/dividery;
  mausymax=iYOffset+20000/dividery;
 }
 iX1clip+=iXOffset;
 iY1clip+=iYOffset;
 iX2clip+=iXOffset;
 iY2clip+=iYOffset;

 /**************************************/
 /* Initialise Line store & hash table */
 /**************************************/
 vector_undraw();
 for(loop=0;loop<LINESTORE_DEPTH-1;loop++)
 {
  line_store[loop].next_vector=loop+1;
  hash_store[loop].next_hash=loop+1;
 }
 line_store[loop].next_vector=-1;
 hash_store[loop].next_hash=-1;

 line_freelist=0;
 hash_freelist=0;

 line_rootnode=-1;

 for(loop=0;loop<HASH_TABLE_SIZE;loop++)
 {
  hash_rootnodes[loop]=-1;
 }
 if (bGblDisplayAutoRefresh)
  wGblDisplayLinePeriod=1;

 if (bGblGUIEnable==T_TRUE)
 {
  hide_all_windows();
 }
 else
 {
  if (display_init_back==T_TRUE)
  {
   hide_all_windows();
  }
 }
//05 fDisplayClear();
 /***********************************************************/
 /* Now screen is reset then reload the Overlay if required */
 /***********************************************************/
 fDNoPicload=T_TRUE;
 fDisplayLoadVOL();
 fDNoPicload=T_FALSE;
 if (wGblImagerMode!=0)
 {
  /* initialise degrees of color wheel in system ticks */
  /* in lwGblWheelTicks is 1/2 turn of the wheel in ticks */
  /* wGblColor1Degrees+wGblColor2Degrees+wGblColor3Degrees should be 180 degrees */
  T_ULONG ltmp_all=wGblColor1Degrees+wGblColor2Degrees+wGblColor3Degrees;
  wGblColor1Degrees = (lwGblWheelTicks*wGblColor1Degrees)/ltmp_all;
  wGblColor2Degrees = (lwGblWheelTicks*wGblColor2Degrees)/ltmp_all;
  wGblColor3Degrees = (lwGblWheelTicks*wGblColor3Degrees)/ltmp_all;
  if (lwGblWheelTicks!=(wGblColor1Degrees+wGblColor2Degrees+wGblColor3Degrees))
  {
   wGblColor1Degrees+=lwGblWheelTicks-(wGblColor1Degrees+wGblColor2Degrees+wGblColor3Degrees);
  }
  if (wGblImagerMode==6)
  {
   if (bGblDisplayExactScaling==T_FALSE)
   {
    wGblImagerXOffset1=-(iXOffset+(-30000>>iDivShiftAx) + (-30000>>iDivShiftBx));
    wGblImagerXOffset2=screen.iXpos-((30000>>iDivShiftAx) + (30000>>iDivShiftBx))-iXOffset ;
    wGblImagerXOffset1=(wGblImagerXOffset1<<iDivShiftAx)+(wGblImagerXOffset1<<iDivShiftBx);
    wGblImagerXOffset2=(wGblImagerXOffset2<<iDivShiftAx)+(wGblImagerXOffset2<<iDivShiftBx);

   }
   else
   {
    wGblImagerXOffset1=-(iXOffset+(-30000/dividerx));
    wGblImagerXOffset2=screen.iXpos-(30000/dividerx)-iXOffset;
    wGblImagerXOffset1*=dividerx;
    wGblImagerXOffset2*=dividerx;

   }
  }
 }

 fIOSetClipping(iX1clip,iY1clip,iX2clip,iY2clip);
 if (bGblGUIEnable==T_TRUE)
 {
//  vector_undraw();
//05  back_save(FORCED_SAVE);
  show_all_windows();
 }
 else
 {
  if (display_init_back==T_TRUE)
  {
//  vector_undraw();
//05   back_save(FORCED_SAVE);
   show_all_windows();
  }
 }
 display_init_back=T_FALSE;

// no need to redo BACK... since we allways load the cart which is loaded
// anyway (hopefully)

/***/

 for (i=12*4096+8*256;i<13*4096;i++)
  fread((T_CHAR *)(atSystemImage+i),1,1,loadram);

 if (bGblExtendedRam==T_TRUE)          /* Animaction Ram 2K */
 {
  for (i=2*4096;i<2*4096+8*256;i++)
   fread((T_CHAR *)(atSystemImage+i),1,1,loadram);
 }
 /***/
 fread(&lXSampleHold,1,sizeof(lXSampleHold),loadram);
 fread(&lYSampleHold,1,sizeof(lYSampleHold),loadram);
 fread(&tZSampleHold,1,sizeof(tZSampleHold),loadram);

 fread(&lSIGDEF_XOUTPUT,1,sizeof(lSIGDEF_XOUTPUT),loadram);
 fread(&lSIGDEF_YOUTPUT,1,sizeof(lSIGDEF_YOUTPUT),loadram);
 fread(&tSIGDEF_ZOUTPUT,1,sizeof(tSIGDEF_ZOUTPUT),loadram);
 /***/

 /***/
 fread(&t6522ORB_0,1,sizeof(t6522ORB_0),loadram);
 fread(&t6522IRB_0,1,sizeof(t6522IRB_0),loadram);

 fread(&t6522IRB_0L, 1, sizeof( t6522IRB_0L), loadram);

 fread(&t6522ORB_1,1,sizeof(t6522ORB_1),loadram);
 fread(&t6522IRB_1,1,sizeof(t6522IRB_1),loadram);
 fread(&t6522IRB_1L,1,sizeof(t6522IRB_1L),loadram);

 fread(&t6522ORB_2,1,sizeof(t6522ORB_2),loadram);
 fread(&t6522IRB_2,1,sizeof(t6522IRB_2),loadram);
 fread(&t6522IRB_2L,1,sizeof(t6522IRB_2L),loadram);

 fread(&t6522ORB_3,1,sizeof(t6522ORB_3),loadram);
 fread(&t6522IRB_3,1,sizeof(t6522IRB_3),loadram);
 fread(&t6522IRB_3L,1,sizeof(t6522IRB_3L),loadram);

 fread(&t6522ORB_4,1,sizeof(t6522ORB_4),loadram);
 fread(&t6522IRB_4,1,sizeof(t6522IRB_4),loadram);
 fread(&t6522IRB_4L,1,sizeof(t6522IRB_4L),loadram);

 fread(&t6522ORB_5,1,sizeof(t6522ORB_5),loadram);
 fread(&t6522IRB_5,1,sizeof(t6522IRB_5),loadram);
 fread(&t6522IRB_5L,1,sizeof(t6522IRB_5L),loadram);

 fread(&t6522ORB_6,1,sizeof(t6522ORB_6),loadram);
 fread(&t6522IRB_6,1,sizeof(t6522IRB_6),loadram);
 fread(&t6522IRB_6L,1,sizeof(t6522IRB_6L),loadram);

 fread(&t6522ORB_7REG,1,sizeof(t6522ORB_7REG),loadram);
 fread(&t6522ORB_7,1,sizeof(t6522ORB_7),loadram);
 fread(&t6522IRB_7,1,sizeof(t6522IRB_7),loadram);
 fread(&t6522IRB_7L,1,sizeof(t6522IRB_7L),loadram);

 fread(&t6522ORA,1,sizeof(t6522ORA),loadram);
 fread(&t6522IRA,1,sizeof(t6522IRA),loadram);
 fread(&t6522IRAL,1,sizeof(t6522IRAL),loadram);

 fread(&t6522DDRB,1,sizeof(t6522DDRB),loadram);
 fread(&t6522DDRA,1,sizeof(t6522DDRA),loadram);

 fread(&t6522SHFTREG,1,sizeof(t6522SHFTREG),loadram);
 fread(&t6522SHFTREGACT,1,sizeof(t6522SHFTREGACT),loadram);

 fread(&u6522Timer1,1,sizeof(TIMER),loadram);
 fread(&u6522Timer2,1,sizeof(TIMER),loadram);
 fread(&bTimer2OneShot,1,sizeof(T_INT),loadram);

 fread(&t6522AUX_T1CTRL,1,sizeof(t6522AUX_T1CTRL),loadram);
 fread(&t6522AUX_T2CTRL,1,sizeof(t6522AUX_T2CTRL),loadram);
 fread(&t6522AUX_SHFTCTRL,1,sizeof(t6522AUX_SHFTCTRL),loadram);
 fread(&t6522AUX_PA_LE,1,sizeof(t6522AUX_PA_LE),loadram);
 fread(&t6522AUX_PB_LE,1,sizeof(t6522AUX_PB_LE),loadram);

 fread(&t6522PCR_CB2CTRL,1,sizeof(t6522PCR_CB2CTRL),loadram);
 fread(&t6522PCR_CB1CTRL,1,sizeof(t6522PCR_CB1CTRL),loadram);
 fread(&t6522PCR_CA2CTRL,1,sizeof(t6522PCR_CA2CTRL),loadram);
 fread(&t6522PCR_CA1CTRL,1,sizeof(t6522PCR_CA1CTRL),loadram);

 fread(&t6522IER,1,sizeof(t6522IER),loadram);

 fread(&t6522IFLG,1,sizeof(t6522IFLG),loadram);

 fread(&t6522CA1,1,sizeof(t6522CA1),loadram);
 fread(&t6522CA2,1,sizeof(t6522CA2),loadram);
 fread(&t6522CB1,1,sizeof(t6522CB1),loadram);
 fread(&t6522CB2,1,sizeof(t6522CB2),loadram);

 fread(&tsCPURegisters,1,sizeof(tsCPURegisters),loadram);
 fread(&iToggle,1,sizeof(iToggle),loadram);
 fread(&tcount,1,sizeof(tcount),loadram);
 fread(&t1int,1,sizeof(t1int),loadram);
 /***/

 /***/
 fread(&iInstruction,1,sizeof(iInstruction),loadram);

 fread(&i6809status,1,sizeof(i6809status),loadram);
 fread(&iLastAddress,1,sizeof(iLastAddress),loadram);
 /***/

 /***/
 fread(&tOLD_BC1,1,sizeof(tOLD_BC1),loadram);
 fread(&tOLD_BDIR,1,sizeof(tOLD_BDIR),loadram);
 fread(&tAddr,1,sizeof(tAddr),loadram);
 /***/

 /***/
/*
 fread(&iLineSelected,1,sizeof(iLineSelected),loadram);
 fread(&iLineSelector,1,sizeof(iLineSelector),loadram);
 fread(&lwRefreshMarker,1,sizeof(lwRefreshMarker),loadram);

 for (i=0;i<LINESTORE_DEPTH;i++)
  fread(&(line_store[0])+i*(sizeof(SDLINE)),1,sizeof(SDLINE),loadram);

 fread(&line_freelist,1,sizeof(line_freelist),loadram);
 fread(&line_rootnode,1,sizeof(line_rootnode),loadram);

 fread(hash_rootnodes,HASH_TABLE_SIZE,sizeof(T_SWORD),loadram);
 fread(&hash_freelist,1,sizeof(T_SWORD),loadram);

 for (i=0;i<LINESTORE_DEPTH;i++)
  fread(&(hash_store[0])+i*(sizeof(SHASH)),1,sizeof(SHASH),loadram);
*/
 /***/

 fclose(loadram);
 if (current_path!=NULL)
 {
  free(current_path);
  current_path=NULL;
 }
 if ((bGblGUIEnable==T_TRUE)||(bGblLightpenEnable==T_TRUE))
 {
  MOUSE_ZEIGE
 }
 tKeyPress=0;
}


/*****************************************************************************/
/*                                                                           */
/*   Function    : fSequencerReset()                                         */
/*                                                                           */
/*   Author      : K.W                                                       */
/*                                                                           */
/*   Description : Reset the vectrex, same as pressing the front button      */
/*                                                                           */
/*   Accepts     : WORD key index that caused the call                       */
/*                                                                           */
/*   Returns     : nothing                                                   */
/*                                                                           */
/*****************************************************************************/
T_VOID fSequencerReset(T_UWORD tKeyPress)
{
 tKeyPress=0;
 f6809Reset();
 fDisplayReset();
}

/*****************************************************************************/
/*                                                                           */
/*   Function    : fSequencerReboot()                                        */
/*                                                                           */
/*   Author      : K.W                                                       */
/*                                                                           */
/*   Description : Perform complete machine initialisation                   */
/*                                                                           */
/*   Accepts     : Nothing                                                   */
/*                                                                           */
/*   Returns     : T_INT    T_TRUE=Init OK, T_FALSE=Init Fail                     */
/*                                                                           */
/*****************************************************************************/
T_VOID fSequencerReboot(T_UWORD tKeyPress)
{
 tKeyPress=0;
 fDebug(HIGH,("fSystemReset() - VECTREX HP3000 Emulator system reset"));
 wGblQuitFlag=EMU_RESTART;
 fDisplayReset();
 fSystemRAMLoad(acGblRAMImageFileName);
 fSystemROMLoad(acGblROMImageFileName);
 fCartridgeROMLoad(acGblDefaultCartFileName);
 f6809Reset();
 f6522Reset();
 fSoundReset();
 fAnalogReset();
}
/*****************************************************************************/
/*                                                                           */
/*   Function    : fCarouselSelect()                                         */
/*                                                                           */
/*   Author      : K.W                                                       */
/*                                                                           */
/*   Description : Select and load the relevant carousel game                */
/*                                                                           */
/*   Accepts     : WORD key index that caused the call                       */
/*                                                                           */
/*   Returns     : nothing                                                   */
/*                                                                           */
/*****************************************************************************/
T_VOID fCarouselSelect(T_UWORD tKeyPress)
{
 switch(tKeyPress)
 {
  case KEY_ALT_F1:
      strcpy(acGblDefaultCartFileName,acGblCartCarousel[0]);
      break;
  case KEY_ALT_F2:
      strcpy(acGblDefaultCartFileName,acGblCartCarousel[1]);
      break;
  case KEY_ALT_F3:
      strcpy(acGblDefaultCartFileName,acGblCartCarousel[2]);
      break;
  case KEY_ALT_F4:
      strcpy(acGblDefaultCartFileName,acGblCartCarousel[3]);
      break;
  case KEY_ALT_F5:
      strcpy(acGblDefaultCartFileName,acGblCartCarousel[4]);
      break;
  case KEY_ALT_F6:
      strcpy(acGblDefaultCartFileName,acGblCartCarousel[5]);
      break;
  case KEY_ALT_F7:
      strcpy(acGblDefaultCartFileName,acGblCartCarousel[6]);
      break;
  case KEY_ALT_F8:
      strcpy(acGblDefaultCartFileName,acGblCartCarousel[7]);
      break;
  case KEY_ALT_F9:
      strcpy(acGblDefaultCartFileName,acGblCartCarousel[8]);
      break;
  case KEY_ALT_F10:
      strcpy(acGblDefaultCartFileName,acGblCartCarousel[9]);
      break;
  case KEY_ALT_F11:
      strcpy(acGblDefaultCartFileName,acGblCartCarousel[10]);
      break;
  case KEY_ALT_F12:
      strcpy(acGblDefaultCartFileName,acGblCartCarousel[11]);
      break;
  default:
      break;
 }
 display_init_back=T_TRUE;
 fSequencerReboot(0);
 set_functions();
 wGblQuitFlag=EMU_RESTART;
}

T_VOID fDebugMonitorToggle(T_UWORD tKeyPress)
{
 tKeyPress=0;
 /* Make sure we enter the monitor even if we're tracing or waiting */
 if (bGblDebugMonitorEnable==T_TRUE)
 {
  //show_message("Monitor allready enabled!");
  aktivate_window_id(MonitorWinId);
  return; // only leave from GUI or Monitor command
 }
 if (bGblGUIEnable==T_FALSE)
 {
  fGUIToggle(0);
 }
 if (open_monitor_window()==T_TRUE)
 {
  bGblDebugMonitorEnable=T_TRUE;
  lwGblSystemTickBreakpoint=0;
 }
 else
  bGblDebugMonitorEnable=T_FALSE;
}

T_VOID fDebugToggle(T_UWORD tKeyPress)
{
 time_t dtime;
 static T_CHAR text[300];
 tKeyPress=0;
 if (bGblDebugEnable==T_FALSE)
 {
  if (bGblGUIEnable==T_FALSE)
  {
   fGUIToggle(0);
  }
  if (open_debug_window()==T_TRUE)
  {
   bGblDebugEnable=T_TRUE;
   // do it this way, 'cause set functions has no grip on the function currently running
   dtime=time(NULL);
   sprintf(text,"%s %s","fDebugToggle(): Debug switched ON", ctime(&dtime));
   strtok(text,"\n\0");
   edit_log_add(get_window(DebugWinId),text);
  }
  else
  {
   bGblDebugEnable=T_FALSE;
  }
 }
 else
 {
  bGblDebugEnable=T_FALSE;
  close_debug_window();
 }
}

T_VOID fGUIToggle(T_UWORD tKeyPress)
{
 tKeyPress=0;
 if (bGblGUIEnable==T_FALSE)
 {
  bGblGUIEnable=T_TRUE;
  vector_undraw();
  back_save(FORCED_SAVE);
  show_all_windows();
  set_functions();
  vector_refresh();
  reset_state=T_FALSE;
  wGblQuitFlag=EMU_RESTART;
 }
}

T_VOID fDebugFileToggle(T_UWORD tKeyPress)
{
 tKeyPress=0;
 if(bGblDebugFileEnable==T_FALSE)
 {
  bGblDebugFileEnable=T_TRUE;
  fDebug(HIGH,("fDebugFileToggle() - FILE OUTPUT ENABLED"));
 }
 else
 {
  bGblDebugFileEnable=T_FALSE;
  fDebug(HIGH,("fDebugFileToggle() - FILE OUTPUT DISABLED"));
 }
}

T_VOID fDebugLevelLow(T_UWORD tKeyPress)
{
 tKeyPress=0;
 tGblDebugLevel=LOW;
 fDebug(HIGH,("fDebugLevel() - ALL MESSAGES"));
}

T_VOID fDebugLevelMid(T_UWORD tKeyPress)
{
 tKeyPress=0;
 tGblDebugLevel=MID;
 fDebug(HIGH,("fDebugLevel() - MED & HIGH MESSAGES"));
}

T_VOID fDebugLevelHigh(T_UWORD tKeyPress)
{
 tKeyPress=0;
 tGblDebugLevel=HIGH;
 fDebug(HIGH,("fDebugLevel() - HIGH MESSAGES"));
}

#endif //#ifndef _NO_DEBUG_INFORMATION_
