/*****************************************************************************/
/*                                                                           */
/*   VECTREX HP3000 Software Emulator              Copyright K.Wilkins 1996  */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*   Title      : Vectrex HP3000 Main Source Code Module                     */
/*                                                                           */
/*   File Name  : VECTREX.C                                                  */
/*                                                                           */
/*   Author     : Keith Wilkins / C.S.                                       */
/*                                                                           */
/*   Version    : 1.20                                                       */
/*                                                                           */
/*   Desciption : This fie contains the main() function for the vectrex      */
/*                emulator software. This module does all the system init    */
/*                functions and contains the sequencer function to run the   */
/*                emulator itself.                                           */
/*                                                                           */
/*   Functions  : main()      - Initialisation, start and exit code          */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*   Revision History:                                                       */
/*                                                                           */
/*   Version    Date    Who  Description of changes                          */
/*   -------    ----    ---  ----------------------                          */
/*                                                                           */
/*    0.01    13/03/93  K.W  File created                                    */
/*    1.00    05/08/96  K.W  Public release of DVE V1.0                      */
/*                                                                           */
/*****************************************************************************/

#define VECTREX_C

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include <audio.h>

#include <standard.h>
#include <tools.h>

#include "io.h"
#include "adlibsnd.h"
#include "sealsnd.h"
#include "nonesnd.h"
#include "keyboard.h"
#include "emu6809.h"
#include "dis6809.h"
#include "emu6522.h"
#include "sequence.h"
#include "seq1.h"
#include "seq2.h"
#include "seq3.h"
#include "seq4.h"
#include "seq5.h"
#include "seq6.h"
#include "seq7.h"
#include "seq8.h"
#include "seq9.h"
#include "seq10.h"
#include "seq11.h"
#include "seq12.h"
#include "sound.h"
#include "analog.h"
#include "joystick.h"
#include "display.h"
#include "verror.h"
#include "vdebug.h"
#include "cart_rom.h"
#include "monitor.h"
#include "mon_cmds.h"
#include "sys_rom.h"
#include "sys_ram.h"
#include "vectrex.h"


#ifdef fDebug
 #undef fDebug
#endif
#define fDebug(level,body) ((bGblDebugEnable==T_TRUE && level>=tGblDebugLevel)?(tGblDebugMsgLevel=level),(fDebugXX body):0)
T_INT start_game_file(T_PCHAR filename);
T_INT execute_vload_game_button(PARAMETER *parameter);
T_INT execute_vedit_ini_button(PARAMETER *parameter);
T_INT execute_vconfig_button(PARAMETER *parameter);
T_INT execute_vtoggle_gui_button(PARAMETER *parameter);
T_INT execute_vreboot_button(PARAMETER *parameter);

/* Local function prototypes */
T_INT fGlobalVariableLoad(T_VOID);

/**************************************************************************/
/* Define all of the Global externals, these are ALL defined in VECTREX.C */
/* by the inclusion of this file so do real declarations for vectrex.c &  */
/* therefore only externals for non VECTREX.C modules                     */
/**************************************************************************/

T_INT blinetype=T_NORMAL;

T_INT gblException=T_FALSE;

UPDATE_FUNCTION *vectrex_update=NULL;
T_INT  action_after_execution=V_DO_NOTHING;
T_INT  init_after_execution=T_FALSE;
T_INT  load_var=0;
T_INT  sthwkptch=T_FALSE;

T_UWORD enable_event=0;
T_UWORD wGblPCJoystick=0; // 0 no (then keyboard) 1 = digital, 2 = analolg

T_UWORD wGblseal_audio_device= AUDIO_DEVICE_MAPPER;
T_UWORD wGblDigitalVolume=80;           /* Digital Volume Level */
T_INT   bGblForceVesa12=T_FALSE;
T_INT   bGblForceSingleProcessor=T_FALSE;
T_INT   bGblGoggleGame=T_FALSE;
T_UWORD ImagerColorNow=255;

T_INT   bGblGUIEnable=T_TRUE;
T_ULONG imager_pulse_ticks=0;

T_ULONG lwGblWheelTicks=(1500000/25)/2; // every 25 ms is default
T_INT   reset_state=T_TRUE;

T_UWORD wGblColor1Degrees=60;
T_UWORD wGblColor2Degrees=60;
T_UWORD wGblColor3Degrees=60;
T_UWORD wGblImagerMode=0;

T_SWORD wGblImagerXOffset1=0;
T_SWORD wGblImagerXOffset2=0;
T_SWORD wGblImagerXOffset=0;

/*******************/
/* Debug Variables */
/*******************/
T_INT   bGblDebugEnable=T_TRUE;           /* Debug prSWORDing screen on/off */
T_INT   bGblDebugFileEnable=T_TRUE;       /* Switch the debug file save on/off */
T_SWORD tGblDebugLevel=HIGH;              /* Debug window display level */
T_SWORD tGblDebugMsgLevel=HIGH;           /* Debug level of the individial messages */
T_INT   bGblDebugMonitorEnable=T_TRUE;    /* Switch the Monitor window on/off*/
T_INT pause_on_monitor_entry_default=T_TRUE;

/***********************/
/* Sequencer variables */
/***********************/

T_INT   bGblExtendedRam=T_FALSE;          /* Animaction Ram 2K */
T_INT   bGblNoShades=T_FALSE;
T_INT   bGblLoadColorsExact=T_FALSE;

T_UWORD wGblQuitFlag=EMU_RUN;             /* The exit flag */

T_INT   bGblStatsValid=T_TRUE;            // Allow stats printing

T_ULONG lwGblSystemTickTotal=1;         /* Total number of SystemTicks since start */
T_ULONG lwGblSystemTickBreakpoint=0;    /* BreakpoSWORD counter */

T_INT   bGblDisplayEnable=T_TRUE;         /* Monitor Persistance enable */
T_INT   bGblDisplayAutoRefresh=T_TRUE;
T_INT   bGblDisplayEnhance=T_FALSE;       // Enable vector repairing on redraw
T_INT   bGblDisplayVerbose=T_FALSE;              /* verbosity of output */
T_INT   bGblDisplayExactScaling=T_FALSE;         /* scaling by SHIFT or DIV */
T_INT   bGblDisplayFullRefresh=T_FALSE; /* set in *.vol refresh everything even if not changed */


T_SLONG lwGblDisplayTickReload=30000;    /* Number of System ticks for one Peristance tick */
T_ULONG lwGblDisplayTickMax=60000;
T_ULONG lwGblDisplayTickMin=7500;
T_SLONG lwGblDisplayTickCounter=1;
T_UWORD wGblDisplayLinePeriod=1;        // How many display ticks a line lives for
T_UWORD wGblDisplayBaseColour=1;

T_INT   bGblOverlayEnable=T_FALSE;
T_INT   bGblOverlayEnable_reset=T_FALSE;
T_INT   bGblLightpenEnable=T_FALSE;       // read in overlay file *.vol

T_SLONG lwGblClipX=17000;               // Clipping window top right
T_SLONG lwGblClipY=18000;
T_SLONG lwGblClipX_reset;
T_SLONG lwGblClipY_reset;

T_UWORD wGblSound=SOUND_ADLIB;          /* Enable the sound system */

T_SLONG lwGblMaxTicks;                  /* Vectrex speed limiter = 100%, converted to ticks later */
T_SLONG savspeed=100;                   /* Vectrex speed limiter = 100%, converted to ticks later */

/**************************************/
/* Emulation type variables Variables */
/**************************************/
T_INT   bGblCartridgeROMWriteEnable=T_FALSE;
T_INT   bGblSystemROMWriteEnable=T_FALSE;

/**********************/
/* Setup default keys */
/**********************/
T_UWORD wGblPly1Button1=SCAN_A;
T_UWORD wGblPly1Button2=SCAN_S;
T_UWORD wGblPly1Button3=SCAN_D;
T_UWORD wGblPly1Button4=SCAN_F;

T_UWORD wGblPly1JoystkUp=SCAN_UP;
T_UWORD wGblPly1JoystkDn=SCAN_DOWN;
T_UWORD wGblPly1JoystkLf=SCAN_LEFT;
T_UWORD wGblPly1JoystkRt=SCAN_RIGHT;

T_UWORD wGblPly2Button1=0x70;
T_UWORD wGblPly2Button2=0x70;
T_UWORD wGblPly2Button3=0x70;
T_UWORD wGblPly2Button4=0x70;

T_UWORD wGblPly2JoystkUp=0x70;
T_UWORD wGblPly2JoystkDn=0x70;
T_UWORD wGblPly2JoystkLf=0x70;
T_UWORD wGblPly2JoystkRt=0x70;

T_INT   bGblDigitalJoystick=T_TRUE;
T_INT   bGblJoystickAutoCentre=T_TRUE;
T_SWORD wGblJoystickSensitivity=2;

T_ULONG lwGblTicksToGrab=1500000/1000*33; // every 33 ms

T_INT   bGblRestoreOnStart=T_TRUE;
T_INT   bGblSaveOnExit=T_TRUE;

T_INT debug_state=T_RELEASE;
T_INT bGblPauseMode=T_FALSE;

/*********************/
/* Define Video mode */
/*********************/
T_UWORD wGblDisplayMode=0;

T_SWORD        wGblXDrift=0;
T_SWORD        wGblYDrift=0;
T_SWORD        wGblXIntegrator=0;
T_SWORD        wGblYIntegrator=0;


/********************/
/* Define filenames */
/********************/
T_UCHAR   acGblROMImageFileName[STRING_LENGTH]="system.img";
T_UCHAR   acGblRAMImageFileName[STRING_LENGTH]="";
T_UCHAR   acGblDefaultCartFileName[STRING_LENGTH]="";
T_UCHAR   acGblCartCarousel[12][STRING_LENGTH]={"","","","","","","","","","","",""};
T_UCHAR   GblBinDir[STRING_LENGTH];
T_UCHAR   GblVolDir[STRING_LENGTH];
T_UCHAR   GblPicDir[STRING_LENGTH];
T_UCHAR   GblSvgDir[STRING_LENGTH];
T_UCHAR   GblDevDir[STRING_LENGTH];

T_VOID  init_vectrex_gui(T_VOID);

T_VOID update_vemu(PARAMETER *parameter)
{
 if (init_after_execution==T_TRUE)
 {
  init_after_execution=T_FALSE;
  switch (action_after_execution)
  {
   case V_DO_NOTHING:
   {
    break;
   }
   case V_LOAD_CARTRIDGE:
   {
    display_init_back=T_TRUE;
    fSequencerReboot(0);
    set_functions();
    wGblQuitFlag=EMU_RUN;
    break;
   }
   case V_LOAD_SAVED_GAME:
   {
    fLoadState(load_var);
    break;
   }
   case V_QUIT:
   {
    close_all_windows();
    wGblQuitFlag=EMU_DONE;
    PROGRAMM_ENDE
    return;
   }
   case V_REBOOT:
   {
    close_all_windows();
    wGblQuitFlag=EMU_DONE;
    PROGRAMM_ENDE
    return;
   }
  }
 }
 if (wGblQuitFlag!=EMU_RUN)
 {
  if (dispatcher_ebene!=1)
  {
   wGblQuitFlag=EMU_RUN;
   if (bGblGUIEnable!=T_TRUE)
   {
    DoneTimerHandler();
    while (bGblGUIEnable!=T_TRUE)
    {
     fVectrexEmulate();
     if (bGblGUIEnable!=T_TRUE)
     {
      if (wGblQuitFlag==EMU_RESTART)
       wGblQuitFlag=EMU_RUN;
      else
       if (wGblQuitFlag==EMU_DONE)
       {
        PROGRAMM_ENDE
        return;
       }
     }
    }
    InitTimerHandler();
    reset_state=T_FALSE;
    firstrun();
    if (wGblQuitFlag==EMU_DONE)
    {
     PROGRAMM_ENDE
     return;
    }
   }
  }
  else
  {
   PROGRAMM_ENDE
   return;
  }
 }
 if (bGblPauseMode==T_FALSE)
  fVectrexEmulate();
 else
  vector_refresh();
}

/*****************************************************************************/
/*                                                                           */
/*   Function   : main()                                                     */
/*                                                                           */
/*   Author     : K.W                                                        */
/*                                                                           */
/*   Desciption : Main initialisation, starup and exit code                  */
/*                                                                           */
/*   Accepts    : argc,argv values for command line args                     */
/*                                                                           */
/*   Returns    : nothing                                                    */
/*                                                                           */
/*****************************************************************************/

T_VOID main(T_SWORD argc,T_UCHAR *argv[])
{
 T_INT     bArgumentFail=T_FALSE;
 T_DOUBLE  dCycSec;
 T_DOUBLE  dEfficiency;
 T_INT     gui_first_time=T_TRUE;
 time_t    tExecTime;

 t_printf(E_VERBOSE_NONE,"DVE version:%s, compiled %s\n",DVE_VERSION,__DATE__);
 srand(123);
 /*******************************/
 /* Decode the arguments passed */
 /*******************************/
 if (argc==2)
 {
  if (strstr(argv[1],"sealinfo"))
  {
   sealinfo();
   return;
  }
 }
 e_verbose=E_VERBOSE_HIGH;
 function_history=T_FALSE;
 current_error_level=85;
 protocol_variant=ERROR_NUMBER_HIGHER;
 protocol_variant=ERROR_NUMBER_EXACT;
 protocol_variant=ERROR_SPECIFIC;
 init_error();

 vesa_force_bank=T_FALSE;
 use_clip=T_TRUE; /* clip memory in tool lib enablen... */
 init_tools();
 if (tool_error==TOOL_INIT_ERROR)
 {
  tool_exit(TOOL_INIT_ERROR);
  printf("\n Konnte Tool-Library aus irgendeinem Grund nicht initialisieren...\n");
  return;
 }
 /**********************************************************/
 /* First set all defaults, command line can then override */
 /**********************************************************/
 if (fGlobalVariableLoad()==T_FALSE)
  bArgumentFail=T_TRUE;
 if (argc==2)
 {
  if (strstr(argv[1],"vesainfo"))
  {
   vesa_anzeige_daten();
   e_verbose=E_VERBOSE_NONE;
   vesa_setze_modus(wGblDisplayMode);
   vesa_anzeige_modus_daten(vesa_modus_active);
   vesa_alten_modus_setzen();
   fcloseall();
   de_init_tools();
   return;
  }
 }
 if (argc==2)
 {
  strcpy(acGblDefaultCartFileName,argv[1]);
  bGblGUIEnable=T_FALSE; // when file given, than no gui is active!
  bGblDebugMonitorEnable=T_FALSE;
  bGblDebugEnable=T_FALSE;
 }
 /*************************************************/
 /* Convert lwGblMaxTicks to tick count from %age */
 /*************************************************/
 // in lwGblMaxTicks steht zur Zeit die % von original Vectrex
 // in der der emulator laufen soll
 if (lwGblMaxTicks>1000)
  lwGblMaxTicks=1000;  // Ha ha ha ha

 lwGblMaxTicks*=(T_SLONG)(1500000.0/UPDATES_PER_SEC);
 // in lwGblMaxTicks steht jetzt, wieviele
 // systemticks zwischen zwei interrupts liegen sollen
 // um 100% geschwindigkeit zu haben
 // mal die prozent, die angegeben wurden

 lwGblMaxTicks/=100;  // Remove %age
 // in lwGblMaxTicks steht jetzt, wieviele
 // systemticks zwischen zwei interrupts liegen sollen
 // um die geforderten % geschwindigkeit zu bekommen

 /*************************************/
 /* Call the initialisation functions */
 /*************************************/
 if (bArgumentFail==T_TRUE)
 {
  fprintf(stderr,"Command line argument decode failed, aborting emulator.\n");
 }
 else
 {
  if (fIOInit()==T_FALSE)
  {
   fprintf(stderr,"I/O Initialisation failed\n");
  }
  else
  {
   vesa_mouse_zeiger=&vectrex_mousepointer;
   vesa_set_default_mouse_color(schwarz,weiss);
   vesa_mouse_zeiger=&vesa_mouse_zeiger_default;
   vesa_set_default_mouse_color(schwarz,weiss);
   if (fSystemInit()==T_FALSE)
   {
    fDebug(VERROR,("Initialisation sequence failed, aborting emulator."));
   }
   else if (fDebugInit()==T_FALSE)
   {
    fprintf(stderr,"Debug Initialisation failed\n");
   }
   else
   {
    /**************/
    /* Here we go */
    /**************/
    fKBDbuffering(T_FALSE);
    tExecTime=clock();

    /* some GUI stuff done first */
    vectrex_update=insert_update_funktion(update_vemu,10);
    default_tasten_funktion=vdefault_tasten_funktion;
    help_user_function=start_game_file;
    do
    {
     wGblQuitFlag=EMU_RUN;
     if (bGblGUIEnable==T_FALSE)
     {
      fVectrexEmulate();
     }
     else
     {
      if (gui_first_time==T_TRUE)
      {
       gui_first_time=T_FALSE;
       init_vectrex_gui();
       if (bGblDebugMonitorEnable==T_TRUE)
       {
        if (open_monitor_window()==T_TRUE)
        {
         bGblDebugMonitorEnable=T_TRUE;
         lwGblSystemTickBreakpoint=0;
         wGblQuitFlag=EMU_RUN;
        }
        else
         bGblDebugMonitorEnable=T_FALSE;
       }
      }
      InitTimerHandler();
      firstrun();
      PROGRAMM_INIT
      DoneTimerHandler();
     }
    }
    while (wGblQuitFlag==EMU_RESTART);

    /* some GUI stuff done last */
    remove_update_funktion(update_vemu);
    if (wGblQuitFlag==EMU_FAULT)
    {
     fDebug(VERROR,("Emulation exited with error"));
    }
    if (wGblQuitFlag==EMU_DONE)
    {
     fDebug(HIGH,("Emulation completed succesfully"));
     if (bGblSaveOnExit==T_TRUE)
     {
      fSaveState(0);
     }
     fKBDbuffering(T_TRUE);
    }
    tExecTime=clock()-tExecTime;
   }
   /*******************/
   /* Prepare to quit */
   /*******************/

   deinit_dis();
   /*************************************************/
   /* Return the screen and io handlers back to DOS */
   /*************************************************/
   fSoundRestore();
   fIORestore();
  }
 }
 /******************************/
 /* Dump execution speed stats */
 /******************************/
 if (bGblStatsValid)
 {
  if (tExecTime/CLOCKS_PER_SEC)
  {
   dCycSec=(lwGblSystemTickTotal/(tExecTime/CLOCKS_PER_SEC));
   dEfficiency=(dCycSec/1500000.0)*100;
   t_printf(E_VERBOSE_NONE,"\nSpeed rating: %.0f Cycles/Sec",dCycSec);
   t_printf(E_VERBOSE_NONE,"\n              %3.1f%%\n",dEfficiency);
  }
  else
  {
   t_printf(E_VERBOSE_NONE,"\nDVE did not run for long enough to perform speed profile\n");
  }
 }
 write_ini();
 /*******************/
 /* Now lets finish */
 /*******************/
 fcloseall();
 de_init_tools();
 exit(0);
}

/*****************************************************************************/
/*                                                                           */
/*   Function    : fSystemInit()                                             */
/*                                                                           */
/*   Author      : K.W / C.S.                                                */
/*                                                                           */
/*   Description : Perform complete machine initialisation                   */
/*                                                                           */
/*   Accepts     : Nothing                                                   */
/*                                                                           */
/*   Returns     : T_INT    T_TRUE=Init OK, T_FALSE=Init Fail                     */
/*                                                                           */
/*****************************************************************************/
T_INT fSystemInit(T_VOID)
{
 T_INT  bInitStatus=T_TRUE;
 fDebug(HIGH,("fSystemReset() - VECTREX HP3000 Emulator Initialisation Sequence"));

 /***********/
 /* Display */
 /***********/
 if ((bInitStatus&=fDisplayInit())==T_FALSE)
 {
  fFatal("fSystemInit() - Vector Display : Initialisation Failed.");
 }
 else
 {
  fDebug(HIGH ,("Vector Display       : O.K."));
 }
 set_functions(); // to do the *.col inits right
 /************/
 /* 6809 CPU */
 /************/
 if ((bInitStatus=f6809Init())==T_FALSE)
 {
  fFatal("fSystemInit() - 68A09 CPU : Initialisation Failed.");
 }
 else
 {
  fDebug(HIGH ,("68A09 CPU            : O.K."));
 }
 /*****************/
 /* 2114 1Kx8 RAM */
 /*****************/
 if ((bInitStatus&=fSystemRAMLoad(acGblRAMImageFileName))==T_FALSE)
 {
  fFatal("fSystemInit() - 2114 RAM : Initialisation Failed.");
 }
 else
 {
  fDebug(HIGH ,("2114 RAM             : O.K."));
 }
 /*****************/
 /* 2363 8Kx8 ROM */
 /*****************/
 if ((bInitStatus&=fSystemROMLoad(acGblROMImageFileName))==T_FALSE)
 {
  fFatal("fSystemInit() - 2363 System ROM Load : Initialisation Failed.");
 }
 else
 {
  fDebug(HIGH ,("2363 System ROM Load : O.K."));
 }
 /*****************/
 /* Cartridge ROM */
 /*****************/
 if ((bInitStatus&=fCartridgeROMLoad(acGblDefaultCartFileName))==T_FALSE)
 {
  fFatal("fSystemInit() - Cartridge ROM Load : Initialisation Failed.");
 }
 else
 {
  fDebug(HIGH ,("Cartridge ROM Load   : O.K."));
 }
 /*************/
 /* 6522A PIA */
 /*************/
 if ((bInitStatus&=f6522Init())==T_FALSE)
 {
  fFatal("fSystemInit() - 6522A VIA : Initialisation Failed.");
 }
 else
 {
  fDebug(HIGH ,("6522A PIA            : O.K."));
 }
 /********************/
 /* AY-3-8192A Sound */
 /********************/

 if ((bInitStatus&=fSoundInit())==T_FALSE)
 {
  wGblSound=SOUND_NONE;
  if ((bInitStatus&=fSoundInit())==T_FALSE)
  {
   fFatal("fSystemInit() - AY-3-8192 Chip : Initialisation Failed.");
  }
  else
  {
   fDebug(HIGH ,("AY-3-8192 Chip       : O.K. (no sound)"));
  }
 }
 else
 {
  fDebug(HIGH ,("AY-3-8192 Chip       : O.K."));
 }
 /*********************/
 /*  Analog functions */
 /*********************/
 if ((bInitStatus&=fAnalogInit())==T_FALSE)
 {
  fFatal("fSystemInit() - Analog Block : Initialisation Failed.");
 }
 else
 {
  fDebug(HIGH ,("Analog Block         : O.K."));
 }
 return(bInitStatus);
}

/*****************************************************************************/
/*                                                                           */
/*   Function    : fGlobalVariableLoad()                                     */
/*                                                                           */
/*   Author      : K.W / C.S.                                                */
/*                                                                           */
/*   Description : Load/Initialise ALL of the global variables.              */
/*                                                                           */
/*   Accepts     : Char* pointer to file to load from                        */
/*                                                                           */
/*   Returns     : T_INT    T_TRUE=Init OK, T_FALSE=Init Fail                     */
/*                                                                           */
/*****************************************************************************/

T_INT fGlobalVariableLoad(T_VOID)
{
 PARSER_WORD parse_hilfe[]=
 {
  {PARSER_TYPE_UWORD   ,"DIGITAL_VOLUME"        ,&wGblDigitalVolume,            0},
  {PARSER_TYPE_INT     ,"FORCE_VESA12"          ,&bGblForceVesa12,              0},
  {PARSER_TYPE_INT     ,"FORCE_SINGLE_PROCESSOR",&bGblForceSingleProcessor,     0},
  {PARSER_TYPE_UWORD   ,"PCJOYSTICK"            ,&wGblPCJoystick,               0},
  {PARSER_TYPE_INT     ,"GUI_ENABLE"            ,&bGblGUIEnable,                0},
  {PARSER_TYPE_INT     ,"DEBUG_ENABLE"          ,&bGblDebugEnable,              0},
  {PARSER_TYPE_INT     ,"DEBUG_MONITOR"         ,&bGblDebugMonitorEnable,       0},
  {PARSER_TYPE_INT     ,"DEBUG_MONITOR_PAUSE"   ,&pause_on_monitor_entry_default,0},
  {PARSER_TYPE_INT     ,"RESTORE_ON_START"      ,&bGblRestoreOnStart,           0},
  {PARSER_TYPE_INT     ,"SAVE_ON_EXIT"          ,&bGblSaveOnExit,               0},
  {PARSER_TYPE_INT     ,"DISPLAY_VERBOSE"       ,&bGblDisplayVerbose,           0},
  {PARSER_TYPE_INT     ,"DISPLAY_EXACT_SCALING" ,&bGblDisplayExactScaling,      0},
  {PARSER_TYPE_INT     ,"DISPLAY_ENABLE"        ,&bGblDisplayEnable,            0},
  {PARSER_TYPE_INT     ,"DISPLAY_AUTO_REFRESH"  ,&bGblDisplayAutoRefresh,       0},
  {PARSER_TYPE_INT     ,"DISPLAY_ENHANCE"       ,&bGblDisplayEnhance,           0},
  {PARSER_TYPE_SLONG   ,"DISPLAY_UPDATE_PERIOD" ,&lwGblDisplayTickReload,       0},
  {PARSER_TYPE_ULONG   ,"DISPLAY_UPDATE_MAX"    ,&lwGblDisplayTickMax,          0},
  {PARSER_TYPE_ULONG   ,"DISPLAY_UPDATE_MIN"    ,&lwGblDisplayTickMin,          0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_LINE_PERIOD"   ,&wGblDisplayLinePeriod,        0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_MODE"          ,&wGblDisplayMode,              0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_COLOUR"        ,&wGblDisplayBaseColour,        0},
  {PARSER_TYPE_SLONG   ,"DISPLAY_CLIP_X"        ,&lwGblClipX,                   0},
  {PARSER_TYPE_SLONG   ,"DISPLAY_CLIP_Y"        ,&lwGblClipY,                   0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_0DIV_X"        ,&(divider_x[0]),               0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_0DIV_Y"        ,&(divider_y[0]),               0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_1DIV_X"        ,&(divider_x[1]),               0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_1DIV_Y"        ,&(divider_y[1]),               0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_2DIV_X"        ,&(divider_x[2]),               0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_2DIV_Y"        ,&(divider_y[2]),               0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_3DIV_X"        ,&(divider_x[3]),               0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_3DIV_Y"        ,&(divider_y[3]),               0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_4DIV_X"        ,&(divider_x[4]),               0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_4DIV_Y"        ,&(divider_y[4]),               0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_5DIV_X"        ,&(divider_x[5]),               0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_5DIV_Y"        ,&(divider_y[5]),               0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_0SHIFTA_X"     ,&(display_infox[0].iDivShiftA),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_0SHIFTB_X"     ,&(display_infox[0].iDivShiftB),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_0SHIFTA_Y"     ,&(display_infoy[0].iDivShiftA),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_0SHIFTB_Y"     ,&(display_infoy[0].iDivShiftB),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_1SHIFTA_X"     ,&(display_infox[1].iDivShiftA),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_1SHIFTB_X"     ,&(display_infox[1].iDivShiftB),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_1SHIFTA_Y"     ,&(display_infoy[1].iDivShiftA),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_1SHIFTB_Y"     ,&(display_infoy[1].iDivShiftB),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_2SHIFTA_X"     ,&(display_infox[2].iDivShiftA),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_2SHIFTB_X"     ,&(display_infox[2].iDivShiftB),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_2SHIFTA_Y"     ,&(display_infoy[2].iDivShiftA),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_2SHIFTB_Y"     ,&(display_infoy[2].iDivShiftB),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_3SHIFTA_X"     ,&(display_infox[3].iDivShiftA),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_3SHIFTB_X"     ,&(display_infox[3].iDivShiftB),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_3SHIFTA_Y"     ,&(display_infoy[3].iDivShiftA),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_3SHIFTB_Y"     ,&(display_infoy[3].iDivShiftB),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_4SHIFTA_X"     ,&(display_infox[4].iDivShiftA),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_4SHIFTB_X"     ,&(display_infox[4].iDivShiftB),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_4SHIFTA_Y"     ,&(display_infoy[4].iDivShiftA),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_4SHIFTB_Y"     ,&(display_infoy[4].iDivShiftB),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_5SHIFTA_X"     ,&(display_infox[5].iDivShiftA),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_5SHIFTB_X"     ,&(display_infox[5].iDivShiftB),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_5SHIFTA_Y"     ,&(display_infoy[5].iDivShiftA),0},
  {PARSER_TYPE_UWORD   ,"DISPLAY_5SHIFTB_Y"     ,&(display_infoy[5].iDivShiftB),0},
  {PARSER_TYPE_INT     ,"OVERLAY_ENABLE"        ,&bGblOverlayEnable,            0},
  {PARSER_TYPE_UWORD   ,"SOUND"                 ,&wGblSound,                    0},
  {PARSER_TYPE_INT     ,"SOUND_ENABLE"          ,&wGblSound,                    0},
  {PARSER_TYPE_UWORD   ,"SEAL_AUDIO_DEVICE"     ,&wGblseal_audio_device,        0},
  {PARSER_TYPE_SLONG   ,"MAX_SPEED"             ,&savspeed,                     0},
  {PARSER_TYPE_INT     ,"ROM_WRITE_ENABLE"      ,&bGblSystemROMWriteEnable,     0},
  {PARSER_TYPE_INT     ,"CART_WRITE_ENABLE"     ,&bGblCartridgeROMWriteEnable,  0},
  {PARSER_TYPE_INT     ,"DIGITAL_JOYSTICK"      ,&bGblDigitalJoystick,          0},
  {PARSER_TYPE_INT     ,"JOYSTICK_AUTO_CENTRE"  ,&bGblJoystickAutoCentre,       0},
  {PARSER_TYPE_SWORD   ,"JOYSTICK_SENSITIVITY"  ,&wGblJoystickSensitivity,      0},
  {PARSER_TYPE_UWORD   ,"PLAYER1BUTTON1"        ,&wGblPly1Button1,              0},
  {PARSER_TYPE_UWORD   ,"PLAYER1BUTTON2"        ,&wGblPly1Button2,              0},
  {PARSER_TYPE_UWORD   ,"PLAYER1BUTTON3"        ,&wGblPly1Button3,              0},
  {PARSER_TYPE_UWORD   ,"PLAYER1BUTTON4"        ,&wGblPly1Button4,              0},
  {PARSER_TYPE_UWORD   ,"PLAYER1UP"             ,&wGblPly1JoystkUp,             0},
  {PARSER_TYPE_UWORD   ,"PLAYER1DOWN"           ,&wGblPly1JoystkDn,             0},
  {PARSER_TYPE_UWORD   ,"PLAYER1LEFT"           ,&wGblPly1JoystkLf,             0},
  {PARSER_TYPE_UWORD   ,"PLAYER1RIGHT"          ,&wGblPly1JoystkRt,             0},
  {PARSER_TYPE_UWORD   ,"PLAYER2BUTTON1"        ,&wGblPly2Button1,              0},
  {PARSER_TYPE_UWORD   ,"PLAYER2BUTTON2"        ,&wGblPly2Button2,              0},
  {PARSER_TYPE_UWORD   ,"PLAYER2BUTTON3"        ,&wGblPly2Button3,              0},
  {PARSER_TYPE_UWORD   ,"PLAYER2BUTTON4"        ,&wGblPly2Button4,              0},
  {PARSER_TYPE_UWORD   ,"PLAYER2UP"             ,&wGblPly2JoystkUp,             0},
  {PARSER_TYPE_UWORD   ,"PLAYER2DOWN"           ,&wGblPly2JoystkDn,             0},
  {PARSER_TYPE_UWORD   ,"PLAYER2LEFT"           ,&wGblPly2JoystkLf,             0},
  {PARSER_TYPE_UWORD   ,"PLAYER2RIGHT"          ,&wGblPly2JoystkRt,             0},
  {PARSER_TYPE_ULONG   ,"TICKSTOGRAB"           ,&lwGblTicksToGrab,             0},
  {PARSER_TYPE_STRING  ,"ROM_IMAGE"             ,&acGblROMImageFileName,        0},
  {PARSER_TYPE_STRING  ,"RAM_IMAGE"             ,&acGblRAMImageFileName,        0},
  {PARSER_TYPE_STRING  ,"DEFAULT_CART"          ,&acGblDefaultCartFileName,     0},
  {PARSER_TYPE_STRING  ,"CARTRIDGE01"           ,&acGblCartCarousel[0],         0},
  {PARSER_TYPE_STRING  ,"CARTRIDGE02"           ,&acGblCartCarousel[1],         0},
  {PARSER_TYPE_STRING  ,"CARTRIDGE03"           ,&acGblCartCarousel[2],         0},
  {PARSER_TYPE_STRING  ,"CARTRIDGE04"           ,&acGblCartCarousel[3],         0},
  {PARSER_TYPE_STRING  ,"CARTRIDGE05"           ,&acGblCartCarousel[4],         0},
  {PARSER_TYPE_STRING  ,"CARTRIDGE06"           ,&acGblCartCarousel[5],         0},
  {PARSER_TYPE_STRING  ,"CARTRIDGE07"           ,&acGblCartCarousel[6],         0},
  {PARSER_TYPE_STRING  ,"CARTRIDGE08"           ,&acGblCartCarousel[7],         0},
  {PARSER_TYPE_STRING  ,"CARTRIDGE09"           ,&acGblCartCarousel[8],         0},
  {PARSER_TYPE_STRING  ,"CARTRIDGE10"           ,&acGblCartCarousel[9],         0},
  {PARSER_TYPE_STRING  ,"CARTRIDGE11"           ,&acGblCartCarousel[10],        0},
  {PARSER_TYPE_STRING  ,"CARTRIDGE12"           ,&acGblCartCarousel[11],        0},
  {PARSER_TYPE_STRING  ,"BINARY_DIRECTORY"      ,&GblBinDir,                    0},
  {PARSER_TYPE_STRING  ,"VOL_FILE_DIRECTORY"    ,&GblVolDir,                    0},
  {PARSER_TYPE_STRING  ,"PICTURE_DIRECTORY"     ,&GblPicDir,                    0},
  {PARSER_TYPE_STRING  ,"SAVE_DIRECTORY"        ,&GblSvgDir,                    0},
  {PARSER_TYPE_STRING  ,"DEV_DIRECTORY"         ,&GblDevDir,                    0},
  {PARSER_TYPE_INT     ,"LINE_TYPE"             ,&blinetype,                    0},
  {PARSER_TYPE_SWORD   ,"X_DRIFT"               ,&wGblXDrift,                   0},
  {PARSER_TYPE_SWORD   ,"Y_DRIFT"               ,&wGblYDrift,                   0},
  {PARSER_TYPE_SWORD   ,"X_INTEGRATOR"          ,&wGblXIntegrator,              0},
  {PARSER_TYPE_SWORD   ,"Y_INTEGRATOR"          ,&wGblYIntegrator,              0},
  {PARSER_TYPE_SCHAR   ,"DAC_OFFSET_POT"        ,&dac_offset,                   0},
  {PARSER_TYPE_SWORD   ,"X_INTEGRATOR_POT"      ,&integrator_offset_x,          0},
  {PARSER_TYPE_SWORD   ,"Y_INTEGRATOR_POT"      ,&integrator_offset_y,          0},
#ifdef ANALOG_CYCLE_PATCH

  {PARSER_TYPE_UWORD   ,"BLANK"      ,&BLANK_REACT_OFFSET_CYCLES,          0},
  {PARSER_TYPE_UWORD   ,"RAMP"      ,&RAMP_REACT_OFFSET_CYCLES,          0},
  {PARSER_TYPE_UWORD   ,"SAMPLE_HOLD"      ,&SAMPLE_HOLD_REACT_OFFSET_CYCLES,          0},
  {PARSER_TYPE_UWORD   ,"DAC"      ,&DAC_REACT_OFFSET_CYCLES,          0},
  {PARSER_TYPE_UWORD   ,"ZERO"      ,&ZERO_REACT_OFFSET_CYCLES,          0},
#endif


  {PARSER_TYPE_END,NULL,NULL,0}
 };
 p_verbosity = 0;
 p_alloc_pointer=T_FALSE;
 p_parse_init();
 strcpy(GblBinDir,".\\bin");
 strcpy(GblVolDir,".\\vol");
 strcpy(GblPicDir,".\\pic");
 strcpy(GblSvgDir,".\\save");
 strcpy(GblDevDir,".\\dev");
 if (bGblDisplayVerbose==T_TRUE)
  t_printf(E_VERBOSE_NONE,"\nIni File loading...");
 if (p_parse("vectrex.ini",parse_hilfe )!=T_OK)
 {
  fFatal("Something wrong in VECTREX.INI!\n");
 }
 else
 {
  if (bGblDisplayVerbose==T_TRUE)
   t_printf(E_VERBOSE_NONE,"\nIni found...");
 }
 lwGblClipX_reset=lwGblClipX;
 lwGblClipY_reset=lwGblClipY;
 bGblOverlayEnable_reset=bGblOverlayEnable;
 divider_x_reset[0]=divider_x[0];
 divider_y_reset[0]=divider_y[0];

 divider_x_reset[1]=divider_x[1];
 divider_y_reset[1]=divider_y[1];

 divider_x_reset[2]=divider_x[2];
 divider_y_reset[2]=divider_y[2];

 divider_x_reset[3]=divider_x[3];
 divider_y_reset[3]=divider_y[3];

 divider_x_reset[4]=divider_x[4];
 divider_y_reset[4]=divider_y[4];

 divider_x_reset[5]=divider_x[5];
 divider_y_reset[5]=divider_y[5];

 display_infox_reset[0].iDivShiftA=display_infox[0].iDivShiftA;
 display_infox_reset[0].iDivShiftB=display_infox[0].iDivShiftB;
 display_infoy_reset[0].iDivShiftA=display_infoy[0].iDivShiftA;
 display_infoy_reset[0].iDivShiftB=display_infoy[0].iDivShiftB;

 display_infox_reset[1].iDivShiftA=display_infox[1].iDivShiftA;
 display_infox_reset[1].iDivShiftB=display_infox[1].iDivShiftB;
 display_infoy_reset[1].iDivShiftA=display_infoy[1].iDivShiftA;
 display_infoy_reset[1].iDivShiftB=display_infoy[1].iDivShiftB;

 display_infox_reset[2].iDivShiftA=display_infox[2].iDivShiftA;
 display_infox_reset[2].iDivShiftB=display_infox[2].iDivShiftB;
 display_infoy_reset[2].iDivShiftA=display_infoy[2].iDivShiftA;
 display_infoy_reset[2].iDivShiftB=display_infoy[2].iDivShiftB;

 display_infox_reset[3].iDivShiftA=display_infox[3].iDivShiftA;
 display_infox_reset[3].iDivShiftB=display_infox[3].iDivShiftB;
 display_infoy_reset[3].iDivShiftA=display_infoy[3].iDivShiftA;
 display_infoy_reset[3].iDivShiftB=display_infoy[3].iDivShiftB;

 display_infox_reset[4].iDivShiftA=display_infox[4].iDivShiftA;
 display_infox_reset[4].iDivShiftB=display_infox[4].iDivShiftB;
 display_infoy_reset[4].iDivShiftA=display_infoy[4].iDivShiftA;
 display_infoy_reset[4].iDivShiftB=display_infoy[4].iDivShiftB;

 display_infox_reset[5].iDivShiftA=display_infox[5].iDivShiftA;
 display_infox_reset[5].iDivShiftB=display_infox[5].iDivShiftB;
 display_infoy_reset[5].iDivShiftA=display_infoy[5].iDivShiftA;
 display_infoy_reset[5].iDivShiftB=display_infoy[5].iDivShiftB;
 if ((bGblDebugMonitorEnable==T_TRUE)||(bGblDebugEnable==T_TRUE))
  bGblGUIEnable=T_TRUE;
 lwGblMaxTicks=savspeed;
 return(T_TRUE);
}

T_VOID set_functions(T_VOID)
{
 T_UWORD loop;
 debug_state=T_RELEASE;
 frstate=0;
 // for now GUI is all DEBUG!!!
 // && ist korrekt
 if ((bGblDebugEnable==T_TRUE)&&(bGblGUIEnable==T_TRUE))
 {
  debug_state=T_DEBUG;
  bGblGUIEnable=T_TRUE;
 }
 fSoundInit=fnoneSoundInit;
 fSoundReset=fnoneSoundReset;
 fSoundRestore=fnoneSoundRestore;
 fSoundTick=fnoneSoundTick;
 fSoundMute=fnoneSoundMute;
 fSoundResume=fnoneSoundResume;
 InitTimerHandler=noneInitTimerHandler;
 DoneTimerHandler=noneDoneTimerHandler;

 set_seq0_functions(debug_state);
 set_analog_functions(debug_state);
 set_adlib_functions(debug_state);
 set_cart_rom_functions(debug_state);
 set_emu6522_functions(debug_state);
 set_emu6809_functions(debug_state);
 set_none_functions(debug_state);
 set_seal_functions(debug_state);
 if (bGblGUIEnable==T_TRUE)
 {
  set_seq1_functions(T_DEBUG);
  set_seq2_functions(T_DEBUG);
  set_seq3_functions(T_DEBUG);
  set_seq4_functions(T_DEBUG);
  set_seq5_functions(T_DEBUG);
  set_seq6_functions(T_DEBUG);
  set_seq7_functions(T_DEBUG);
  set_seq8_functions(T_DEBUG);
  set_seq9_functions(T_DEBUG);
  set_seq10_functions(T_DEBUG);
  set_seq11_functions(T_DEBUG);
  set_seq12_functions(T_DEBUG);
 }
 else
 {
  set_seq1_functions(debug_state);
  set_seq2_functions(debug_state);
  set_seq3_functions(debug_state);
  set_seq4_functions(debug_state);
  set_seq5_functions(debug_state);
  set_seq6_functions(debug_state);
  set_seq7_functions(debug_state);
  set_seq8_functions(debug_state);
  set_seq9_functions(debug_state);
  set_seq10_functions(debug_state);
  set_seq11_functions(debug_state);
  set_seq12_functions(debug_state);
 }

 set_sys_ram_functions(debug_state);
 set_sys_rom_functions(debug_state);

 f6522Addr0DRead=f6522Addr0DReadNoLightpen;

 /********************************************/
 /* first set all specific functions         */
 /* mixture between: SHIFT/DIV/LINEAR/BANKED */
 /* neccessary because of 'inline' functions */
 /********************************************/
 if (wGblPCJoystick==0)
 {
  if (bGblDigitalJoystick==T_TRUE)
   fAnalogJoystickUpdate=fAnalogJoystickUpdateDigital;
  else
   fAnalogJoystickUpdate=fAnalogJoystickUpdateAnalog;
 }
 else
 {
  if (PCJoystick_init()!=T_TRUE)
  {
   fFatal("Joystick can't be set...");
  }
  if (wGblPCJoystick==1)
  {
   fAnalogJoystickUpdate=fAnalogJoystickUpdatePCJoystickDigital;
  }
  else // then 2 is assumed
  {
   fAnalogJoystickUpdate=fAnalogJoystickUpdatePCJoystickAnalog;
  }
 }
 if (framebuffer==FB_LINEAR) /* linear framebuffer */
 {
  if (blinetype==T_NORMAL)
   fVectrexEmulate=fVectrexEmulate_linear_no_lightpen;
  else
   fVectrexEmulate=fVectrexEmulate_linear_no_lightpen_aa;
  if (bGblDisplayExactScaling==T_FALSE) /* SHIFT */
  {
   if (blinetype==T_NORMAL)
    fAnalogTick=fAnalogTick_linear_shift;
   else
    fAnalogTick=fAnalogTick_linear_shift_aa;
  }
  else /* DIV */
  {
   if (blinetype==T_NORMAL)
    fAnalogTick=fAnalogTick_linear_div;
   else
    fAnalogTick=fAnalogTick_linear_div_aa;
  }
 }
 else /* banked framebuffer */
 {
  if (blinetype==T_NORMAL)
   fVectrexEmulate=fVectrexEmulate_banked_no_lightpen;
  else
   fVectrexEmulate=fVectrexEmulate_banked_no_lightpen_aa;
  if (bGblDisplayExactScaling==T_FALSE) /* SHIFT */
  {
   if (blinetype==T_NORMAL)
    fAnalogTick=fAnalogTick_bank_shift;
   else
    fAnalogTick=fAnalogTick_bank_shift_aa;
  }
  else /* DIV */
  {
   if (blinetype==T_NORMAL)
    fAnalogTick=fAnalogTick_bank_div;
   else
    fAnalogTick=fAnalogTick_bank_div_aa;
  }
 }
 if (wGblSound==SOUND_NONE)
 {
  fSoundInit   = fnoneSoundInit    ;
  fSoundReset  = fnoneSoundReset   ;
  fSoundRestore= fnoneSoundRestore ;
  fSoundTick   = fnoneSoundTick    ;
  fSoundMute   = fnoneSoundMute    ;
  fSoundResume = fnoneSoundResume  ;
  InitTimerHandler=noneInitTimerHandler;
  DoneTimerHandler=noneDoneTimerHandler;
 }
 if (wGblSound==SOUND_ADLIB)
 {
  fSoundInit   = fadlibSoundInit    ;
  fSoundReset  = fadlibSoundReset   ;
  fSoundRestore= fadlibSoundRestore ;
  fSoundTick   = fadlibSoundTick    ;
  fSoundMute   = fadlibSoundMute    ;
  fSoundResume = fadlibSoundResume  ;
  InitTimerHandler=adlibInitTimerHandler;
  DoneTimerHandler=adlibDoneTimerHandler;
 }
 if (wGblSound==SOUND_SEAL)
 {
  fSoundInit   = fsealSoundInit    ;
  fSoundReset  = fsealSoundReset   ;
  fSoundRestore= fsealSoundRestore ;
  fSoundTick   = fsealSoundTick    ;
  fSoundMute   = fsealSoundMute    ;
  fSoundResume = fsealSoundResume  ;
  InitTimerHandler=sealInitTimerHandler;
  DoneTimerHandler=sealDoneTimerHandler;
 }
 if (bGblGoggleGame==T_TRUE)
 {
  if (framebuffer==FB_LINEAR) /* linear framebuffer */
  {
   if (bGblDisplayExactScaling==T_FALSE) /* SHIFT */
   {
    if (blinetype==T_NORMAL)
     fAnalogTick=fAnalogTick_linear_shift_imager;
    else
     fAnalogTick=fAnalogTick_linear_shift_imager_aa;
   }
   else /* DIV */
   {
    if (blinetype==T_NORMAL)
     fAnalogTick=fAnalogTick_linear_div_imager;
    else
     fAnalogTick=fAnalogTick_linear_div_imager_aa;
   }
  }
  else /* banked framebuffer */
  {
   if (bGblDisplayExactScaling==T_FALSE) /* SHIFT */
   {
    if (blinetype==T_NORMAL)
     fAnalogTick=fAnalogTick_bank_shift_imager;
    else
     fAnalogTick=fAnalogTick_bank_shift_imager_aa;
   }
   else /* DIV */
   {
    if (blinetype==T_NORMAL)
     fAnalogTick=fAnalogTick_bank_div_imager;
    else
     fAnalogTick=fAnalogTick_bank_div_imager_aa;
   }
  }
  if (framebuffer==FB_LINEAR) /* linear framebuffer */
  {
   if (blinetype==T_NORMAL)
    fVectrexEmulate=fVectrexEmulate_linear_imager;
   else
    fVectrexEmulate=fVectrexEmulate_linear_imager_aa;
  }
  else /* banked framebuffer */
  {
   if (blinetype==T_NORMAL)
    fVectrexEmulate=fVectrexEmulate_banked_imager;
   else
    fVectrexEmulate=fVectrexEmulate_banked_imager_aa;
  }
 }
 if (bGblLightpenEnable==T_TRUE)
 {
  f6522Addr0DRead=f6522Addr0DReadLightpen;
  if (framebuffer==FB_LINEAR) /* linear framebuffer */
  {
   if (blinetype==T_NORMAL)
    fVectrexEmulate=fVectrexEmulate_linear_lightpen;
   else
    fVectrexEmulate=fVectrexEmulate_linear_lightpen_aa;
   if (bGblDisplayExactScaling==T_FALSE) /* SHIFT */
   {
    if (blinetype==T_NORMAL)
     fAnalogTick=fAnalogTick_linear_shift_lightpen;
    else
     fAnalogTick=fAnalogTick_linear_shift_lightpen_aa;
   }
   else /* DIV */
   {
    if (blinetype==T_NORMAL)
     fAnalogTick=fAnalogTick_linear_div_lightpen;
    else
     fAnalogTick=fAnalogTick_linear_div_lightpen_aa;
   }
  }
  else /* banked framebuffer */
  {
   if (blinetype==T_NORMAL)
    fVectrexEmulate=fVectrexEmulate_banked_lightpen;
   else
    fVectrexEmulate=fVectrexEmulate_banked_lightpen_aa;
   if (bGblDisplayExactScaling==T_FALSE) /* SHIFT */
   {
    if (blinetype==T_NORMAL)
     fAnalogTick=fAnalogTick_bank_shift_lightpen;
    else
     fAnalogTick=fAnalogTick_bank_shift_lightpen_aa;
   }
   else /* DIV */
   {
    if (blinetype==T_NORMAL)
     fAnalogTick=fAnalogTick_bank_div_lightpen;
    else
     fAnalogTick=fAnalogTick_bank_div_lightpen_aa;
   }
  }
 }
 if (vio_initialised==T_TRUE)
 {
  vesa_mouse_deinit();
  if (bGblLightpenEnable==T_TRUE)
  {
   use_vectrex_mouse=T_TRUE;
   vesa_mouse_zeiger=&vectrex_mousepointer;
   vesa_mouse_init();
  }
  else
  {
   if (bGblGUIEnable==T_TRUE)
   {
    use_vectrex_mouse=T_TRUE;
    vesa_mouse_zeiger=&vesa_mouse_zeiger_default;
    vesa_mouse_init();
   }
   else
   {
    use_vectrex_mouse=T_FALSE;
    vesa_mouse_zeiger=&vesa_mouse_zeiger_default;
   }
  }
 }
 // the 0D read function have to be reset to be able to emulate lightpen
 // games when switched on the fly!
 for(loop=0;loop<128;loop++)
 {
  f6809DeclareAddress((loop*16)+PIA6522_ADDR0D,1,f6522Addr0DRead,f6522Addr0DWrite);
 }
 pause_on_monitor_entry=pause_on_monitor_entry_default;
}

T_INT write_ini(T_VOID)
{
 FILE *sysfile;
 T_INT i;
 sysfile=fopen("vectrex.ini", "w");
 if (sysfile==NULL)
 {
  t_printf(E_VERBOSE_NONE,"Debug Initialisation failed\n");
  return T_NOT_OK;
 }
 fprintf(sysfile,"# This is a file for DVE, the Dos Vectrex Emulator Version %s.\n",DVE_VERSION);
 fprintf(sysfile,"# It is automatically created each time DVE is executed and quitted \n# correctly.\n");
 fprintf(sysfile,"# Following are all valid options for this file and a short\n# describtion is given.\n");
 fprintf(sysfile,"\n# The Variables are of different kind, I will provide some information\n# what kind of variable each is.\n");
 fprintf(sysfile,"# There are: \n");
 fprintf(sysfile,"#          : integer (they are treated as boolean, 1 or 0) \n");
 fprintf(sysfile,"#          : unsigned long integer \n");
 fprintf(sysfile,"#          : signed long integer \n");
 fprintf(sysfile,"#          : unsigned short integer \n");
 fprintf(sysfile,"#          : signed short integer \n");
 fprintf(sysfile,"#          : unsigned short character \n");
 fprintf(sysfile,"#          : signed short character \n");
 fprintf(sysfile,"#          : string \n");
 fprintf(sysfile,"# the lengths of them are compiler specific, but I'll give examples\n# of correct values.\n");

 fprintf(sysfile,"\n# X_DRIFT (signed short integer), values - xx to +xx\n");
 fprintf(sysfile,"X_DRIFT                         :%i\n",wGblXDrift);
 fprintf(sysfile,"\n# Y_DRIFT (signed short integer), values - xx to +xx\n");
 fprintf(sysfile,"Y_DRIFT                         :%i\n",wGblYDrift);

 fprintf(sysfile,"\n# X_INTEGRATOR (signed short integer), values - xx to +xx\n");
 fprintf(sysfile,"X_INTEGRATOR                    :%i\n",wGblXIntegrator);
 fprintf(sysfile,"\n# Y_INTEGRATOR (signed short integer), values - xx to +xx\n");
 fprintf(sysfile,"Y_INTEGRATOR                    :%i\n",wGblYIntegrator);

 fprintf(sysfile,"\n# X_INTEGRATOR_POT (signed short integer), values - xx to +xx\n");
 fprintf(sysfile,"X_INTEGRATOR_POT                :%i\n",integrator_offset_x);
 fprintf(sysfile,"\n# Y_INTEGRATOR_POT (signed short integer), values - xx to +xx\n");
 fprintf(sysfile,"Y_INTEGRATOR_POT                :%i\n",integrator_offset_y);

 fprintf(sysfile,"\n# DAC_OFFSET_POT (signed char), values - xx to +xx\n");
 fprintf(sysfile,"DAC_OFFSET_POT                  :%i\n",dac_offset);

 fprintf(sysfile,"\n# Let us first look at the main options, which have to do\n# with display settings.\n");

 fprintf(sysfile,"\n# DISPLAY_ENABLE (integer), values 0 or 1\n");
 fprintf(sysfile,"# Lets you controll whether you want to see any output (1) or\n");
 fprintf(sysfile,"# not (0). For the majority of you this option is very useless.\n");
 fprintf(sysfile,"# Lets allways have a 1 here!\n");
 fprintf(sysfile,"DISPLAY_ENABLE                  :%i\n",bGblDisplayEnable);

 fprintf(sysfile,"\n# DISPLAY_MODE (signed short integer), values 0, 1, 2, 3, 4, 5 \n");
 fprintf(sysfile,"# This should be an unsigned variable, well, doesn't really matter.\n");
 fprintf(sysfile,"# This one lets you chose what resolution you want the \n");
 fprintf(sysfile,"# emulator to use.\n");
 fprintf(sysfile,"# All modes used are VESA modes.\n");
 fprintf(sysfile,"# If a linear framebuffer capable card is found, it is used.\n");
 fprintf(sysfile,"# (which results in a performance boost)\n");
 fprintf(sysfile,"# ATTENTION!!!\n");
 fprintf(sysfile,"# Incorrect settings may harm you monitor, please be\n");
 fprintf(sysfile,"# sure you know what your monitor is capable of befor\n");
 fprintf(sysfile,"# fiddling with some of the higher settings.\n");
 fprintf(sysfile,"# Here is what each number stands for:\n");
 fprintf(sysfile,"#          : 0 Mode 640x480 Pixel 256 colors \n");
 fprintf(sysfile,"#          : 1 Mode 800x600 Pixel 256 colors \n");
 fprintf(sysfile,"#          : 2 Mode 1024x768 Pixel 256 colors \n");
 fprintf(sysfile,"#          : 3 Mode 1280x1024 Pixel 256 colors \n");
 fprintf(sysfile,"#          : 4 Mode 1600x1200 Pixel 256 colors \n");
 fprintf(sysfile,"#          : 5 Mode 320x200 Pixel 256 colors \n");
 fprintf(sysfile,"# NOTE:\n");
 fprintf(sysfile,"# Mode number 5 is no 'ordinary' VESA mode.\n");
 fprintf(sysfile,"# It is possible that you can only set it, if you are\n");
 fprintf(sysfile,"# using a third party VESA Driver (like the excellent one\n");
 fprintf(sysfile,"# from SCITECH)\n");
 fprintf(sysfile,"DISPLAY_MODE                    :%i\n",wGblDisplayMode);

 fprintf(sysfile,"\n# FORCE_VESA12 (integer), values 0 or 1\n");
 fprintf(sysfile,"# This option relates directly to the option mentioned last.\n");
 fprintf(sysfile,"# There have been times when users had trouble using the VESA modes\n");
 fprintf(sysfile,"# DVE chose to use, perhaps this option helps a bit.\n");
 fprintf(sysfile,"# If it is set to 1 only VESA v1.2 features are used, mainly\n");
 fprintf(sysfile,"# this means disabling linear framebuffer usage.\n");
 fprintf(sysfile,"# If your card does not support VESA 2.0 features anyway, you\n");
 fprintf(sysfile,"# can happyly ignore this feature, as DVE uses the VESA 1.2 routines in\n");
 fprintf(sysfile,"# that case anyway...\n");
 fprintf(sysfile,"FORCE_VESA12                    :%i\n",bGblForceVesa12);

 fprintf(sysfile,"\n# LINE_TYPE (integer), values 0 or 1\n");
 fprintf(sysfile,"# Use normal or anti aliased lines.\n");
 fprintf(sysfile,"# 0 means normal lines.\n");
 fprintf(sysfile,"# 1 means anti aliased lines (only for fast systems).\n");
 fprintf(sysfile,"LINE_TYPE                       :%i\n",blinetype);

 fprintf(sysfile,"\n# DISPLAY_EXACT_SCALING (integer), values 0 or 1\n");
 fprintf(sysfile,"# Since all Vectrex output is done via vectors, the output is\n");
 fprintf(sysfile,"# very easy to scale. Actually this is a neccessity, since the\n");
 fprintf(sysfile,"# coordinates used internally range from somewhere around\n");
 fprintf(sysfile,"# -20000 to +20000. To downscale these coordinates one can use \n");
 fprintf(sysfile,"# different functions. Two methods can be used with the emulator, \n");
 fprintf(sysfile,"# one quite fast method, but not so accurate and one more accurate \n");
 fprintf(sysfile,"# (and slower). Internally (computer internally) this means function1 \n");
 fprintf(sysfile,"# uses only SHIFT instructions and an addition, both can be achieved very \n");
 fprintf(sysfile,"# fast, because it is in the blood of every electric beeing.\n");
 fprintf(sysfile,"# The more exact method uses a (wonder!) DIVIDE instruction (which\n");
 fprintf(sysfile,"# all electric beeing I know of HATE).\n");
 fprintf(sysfile,"DISPLAY_EXACT_SCALING           :%i\n",bGblDisplayExactScaling);

 fprintf(sysfile,"\n# DISPLAY_AUTO_REFRESH (integer), values 0 or 1\n");
 fprintf(sysfile,"# If set, the emulator decides whether it would like to update the\n");
 fprintf(sysfile,"# screen or not.\n");
 fprintf(sysfile,"# Actually I (Chris) have not looked at the code for this option\n");
 fprintf(sysfile,"# yet, so I don't know much about it, I usually leave this to 1.\n");
 fprintf(sysfile,"# Look at the documentation for some further information.\n");
 fprintf(sysfile,"DISPLAY_AUTO_REFRESH            :%i\n", bGblDisplayAutoRefresh);

 fprintf(sysfile,"\n# DISPLAY_UPDATE_MAX (unsigned long integer), values 0 to (big number)\n");
 fprintf(sysfile,"# This option is directly related to the above mentioned DISPLAY_AUTO_REFRESH,\n");
 fprintf(sysfile,"# it sets the upper limit the automatic refresh calculation routine \n");
 fprintf(sysfile,"# uses. Values from 40000 to 90000 are somewhat meaningfull. \n");
 fprintf(sysfile,"# The default is 60000 (wise choice I guess).\n");
 fprintf(sysfile,"# This value should be greater than DISPLAY_UPDATE_MIN.\n");
 fprintf(sysfile,"# The measure is in 'TICKS', one tick relates directly to\n");
 fprintf(sysfile,"# the cycle count used in the emulation. So if you run at 100%%\n");
 fprintf(sysfile,"# speed, you should be able to do (about) 1500000 ticks/second.\n");
 fprintf(sysfile,"# -> see also DISPLAY_UPDATE_PERIOD\n");
 fprintf(sysfile,"DISPLAY_UPDATE_MAX              :%i\n", lwGblDisplayTickMax);

 fprintf(sysfile,"\n# DISPLAY_UPDATE_MIN (unsigned long integer), values 0 to (big number)\n");
 fprintf(sysfile,"# This option is directly related to the above mentioned DISPLAY_AUTO_REFRESH,\n");
 fprintf(sysfile,"# it sets the lower limit the automatic refresh calculation routine \n");
 fprintf(sysfile,"# uses. Values from 5000 to 35000 are somewhat meaningfull. \n");
 fprintf(sysfile,"# The default is 7500 (wise choice I guess, though flickery).\n");
 fprintf(sysfile,"# This value should be lower than DISPLAY_UPDATE_MAX.\n");
 fprintf(sysfile,"# The measure is in 'TICKS', one tick relates directly to\n");
 fprintf(sysfile,"# the cycle count used in the emulation. So if you run at 100%%\n");
 fprintf(sysfile,"# speed, you should be able to do (about) 1500000 ticks/second.\n");
 fprintf(sysfile,"# -> see also DISPLAY_UPDATE_PERIOD\n");
 fprintf(sysfile,"DISPLAY_UPDATE_MIN              :%i\n", lwGblDisplayTickMin);

 fprintf(sysfile,"\n# DISPLAY_UPDATE_PERIOD (unsigned long integer), values 0 to (big number)\n");
 fprintf(sysfile,"# This also gives the start for DISPLAY_AUTO_REFRESH.\n");
 fprintf(sysfile,"# Else it gives the exact number of ticks a line drawn is left\n");
 fprintf(sysfile,"# on the screen.\n");
 fprintf(sysfile,"# The measure is in 'TICKS', one tick relates directly to\n");
 fprintf(sysfile,"# the cycle count used in the emulation. So if you run at 100%%\n");
 fprintf(sysfile,"# speed, you should be able to do (about) 1500000 ticks/second.\n");
 fprintf(sysfile,"# A low value means that vectors are displayed a very short time, and\n");
 fprintf(sysfile,"# they may be gone before the vectrex had time to refresh them by its own\n");
 fprintf(sysfile,"# means. (a flicker will occur)\n");
 fprintf(sysfile,"# A high value on the other side increases the persistence of the vectors,\n");
 fprintf(sysfile,"# if set to high, vector trails begin to appear and emulation slowdown\n");
 fprintf(sysfile,"# may occurs. (if set very high the emulator might even go up in flames,\n");
 fprintf(sysfile,"# since there is a fixed amount of memory allocated for lines,\n");
 fprintf(sysfile,"# for %i lines actually)\n",LINESTORE_DEPTH);
 fprintf(sysfile,"DISPLAY_UPDATE_PERIOD           :%i\n", lwGblDisplayTickReload);

 fprintf(sysfile,"\n# DISPLAY_ENHANCE (integer), values 0 or 1\n");
 fprintf(sysfile,"# Well, unlike other emulators, DVE does not have any buffering\n");
 fprintf(sysfile,"# systems. We draw everything right to the screen, be it new lines\n");
 fprintf(sysfile,"# or deleting old ones. This option influences the behavior of\n");
 fprintf(sysfile,"# deleting the old ones. Since it would be far to time consuming\n");
 fprintf(sysfile,"# to check 'collisions' of lines, the old ones are just deleted.\n");
 fprintf(sysfile,"# This leaves the screen somewhat garbled (no, not all that much).\n");
 fprintf(sysfile,"# The enhancement (if enabled) redraws every line after deleting the\n");
 fprintf(sysfile,"# old lines. This cleans up the display again, at the cost of speed.\n");
 fprintf(sysfile,"# If you system is slow and/or you have a graphic card which only supports\n");
 fprintf(sysfile,"# banked video modes, it might be worth trying without enhancement.\n");
 fprintf(sysfile,"# (on fast computers with fast graphic cards, this option does not make\n");
 fprintf(sysfile,"# any significant speed differences, so you should leave it on)\n");
 fprintf(sysfile,"DISPLAY_ENHANCE                 :%i\n", bGblDisplayEnhance);

 fprintf(sysfile,"\n# DISPLAY_LINE_PERIOD (unsigned short integer), values 0 to ...\n");
 fprintf(sysfile,"# I don't really know what this was actually used for, by now it\n");
 fprintf(sysfile,"# is obsolete!\n");
 fprintf(sysfile,"DISPLAY_LINE_PERIOD             :%i\n", wGblDisplayLinePeriod);

 fprintf(sysfile,"\n# OVERLAY_ENABLE (integer), values 0 or 1\n");
 fprintf(sysfile,"# Well, it says it, doesn't it? The option can be set also in\n");
 fprintf(sysfile,"# the appropriate *.vol file, which override the settings found here.\n");
 fprintf(sysfile,"OVERLAY_ENABLE                  :%i\n", bGblOverlayEnable);

 fprintf(sysfile,"\n# DISPLAY_COLOUR (unsigned short integer), values 1 to 14\n");
 fprintf(sysfile,"# If overlays are disabled you can still chose what color\n");
 fprintf(sysfile,"# to use for vector drawing (or in old/style overlays).\n");
 fprintf(sysfile,"# Colors range from 1 to 14, which are actally pen colors, which\n");
 fprintf(sysfile,"# each extend to 14 shades of the 'real' color for different\n");
 fprintf(sysfile,"# intensities. When enabling the GUI you will probably\n");
 fprintf(sysfile,"# not want to chose color 14, since colors from that pen\n");
 fprintf(sysfile,"# are used for the it.\n");
 fprintf(sysfile,"# The option can be set also in\n");
/*
 1 pen color * 16
 14 pen color * 16
*/
 fprintf(sysfile,"# the appropriate *.vol file, which override the settings found here.\n");
 fprintf(sysfile,"DISPLAY_COLOUR                  :%i\n", wGblDisplayBaseColour);

 fprintf(sysfile,"\n# DISPLAY_CLIP_X (Y) (signed long integer), values 1 to 20000\n");
 fprintf(sysfile,"# If overlays are disabled you can set the clipping boundaries\n");
 fprintf(sysfile,"# With these two variables. It is allways a rectangle clipped, with its\n");
 fprintf(sysfile,"# center in the middle of the screen.\n");
 fprintf(sysfile,"# That mean:\n");
 fprintf(sysfile,"# DISPLAY_CLIP_X 1000\n");
 fprintf(sysfile,"# DISPLAY_CLIP_Y 1000\n");
 fprintf(sysfile,"# would result in only a fairly small visible square in the\n");
 fprintf(sysfile,"# middle of the screen.\n");
 fprintf(sysfile,"DISPLAY_CLIP_X                  :%i\n", lwGblClipX);
 fprintf(sysfile,"DISPLAY_CLIP_Y                  :%i\n", lwGblClipY);

 fprintf(sysfile,"\n# DISPLAY_#DIV_X (Y) (unsigned short integer), values 1 to 200 (?)\n");
 fprintf(sysfile,"# For these settings to take effect you also have to enable\n");
 fprintf(sysfile,"# DISPLAY_EXACT_SCALING. For each supported video mode (0-5)\n");
 fprintf(sysfile,"# you can enter divider for both x and y resolution.\n");
 fprintf(sysfile,"# With these setting you can scale your vectrex display to any\n");
 fprintf(sysfile,"# size you like. The default values give you a pretty accurate\n");
 fprintf(sysfile,"# display, but if you fiddle with some custom overlays it might be\n");
 fprintf(sysfile,"# easier to scale the vectrex output than to scale the overlay with\n");
 fprintf(sysfile,"# some paint program.\n");
 fprintf(sysfile,"# You can fit the vectrex output to the whole screen with\n");
 fprintf(sysfile,"# the appropriate values (though it might look strange)\n");
 fprintf(sysfile,"# Default values:  Mode    X      Y\n");
 fprintf(sysfile,"#                    0     95     92\n");
 fprintf(sysfile,"#                    1     76     76\n");
 fprintf(sysfile,"#                    2     60     60\n");
 fprintf(sysfile,"#                    3     45     45\n");
 fprintf(sysfile,"#                    4     38     38\n");
 fprintf(sysfile,"#                    5     175    222\n");
 fprintf(sysfile,"# The following formular is used:\n");
 fprintf(sysfile,"#                    X=lX/dividerx;\n");
 fprintf(sysfile,"#                    Y=lY/dividery;\n");
 fprintf(sysfile,"# Note:\n");
 fprintf(sysfile,"# These settings can also be specified in the *.vol files,\n");
 fprintf(sysfile,"# which overwrite settings found here!\n");
 fprintf(sysfile,"DISPLAY_0DIV_X                  :%i\n", divider_x[0]);
 fprintf(sysfile,"DISPLAY_0DIV_Y                  :%i\n", divider_y[0]);
 fprintf(sysfile,"DISPLAY_1DIV_X                  :%i\n", divider_x[1]);
 fprintf(sysfile,"DISPLAY_1DIV_Y                  :%i\n", divider_y[1]);
 fprintf(sysfile,"DISPLAY_2DIV_X                  :%i\n", divider_x[2]);
 fprintf(sysfile,"DISPLAY_2DIV_Y                  :%i\n", divider_y[2]);
 fprintf(sysfile,"DISPLAY_3DIV_X                  :%i\n", divider_x[3]);
 fprintf(sysfile,"DISPLAY_3DIV_Y                  :%i\n", divider_y[3]);
 fprintf(sysfile,"DISPLAY_4DIV_X                  :%i\n", divider_x[4]);
 fprintf(sysfile,"DISPLAY_4DIV_Y                  :%i\n", divider_y[4]);
 fprintf(sysfile,"DISPLAY_5DIV_X                  :%i\n", divider_x[5]);
 fprintf(sysfile,"DISPLAY_5DIV_Y                  :%i\n", divider_y[5]);

 fprintf(sysfile,"\n# DISPLAY_#SHIFTA_X DISPLAY_#SHIFTB_X(Y) (unsigned short integer), values 1 to 10 (?)\n");
 fprintf(sysfile,"# For these settings to take effect you also have to disable\n");
 fprintf(sysfile,"# DISPLAY_EXACT_SCALING. For each supported video mode (0-5)\n");
 fprintf(sysfile,"# you can enter two shifter for both x and y resolution.\n");
 fprintf(sysfile,"# With these setting you can scale your vectrex display to any\n");
 fprintf(sysfile,"# size you like. The default values give you a pretty accurate\n");
 fprintf(sysfile,"# display, but if you fiddle with some custom overlays it might be\n");
 fprintf(sysfile,"# easier to scale the vectrex output than to scale the overlay with\n");
 fprintf(sysfile,"# some paint program.\n");
 fprintf(sysfile,"# You can fit the vectrex output to the whole screen with\n");
 fprintf(sysfile,"# the appropriate values (though it might look strange)\n");
 fprintf(sysfile,"# Default values:  Mode    X A   X B     Y A  Y B\n");
 fprintf(sysfile,"#                   0       7     9       7    9\n");
 fprintf(sysfile,"#                   1       7     8       7    8\n");
 fprintf(sysfile,"#                   2       6     9       6    9\n");
 fprintf(sysfile,"#                   3       6     8       6    8\n");
 fprintf(sysfile,"#                   4       6     7       6    7\n");
 fprintf(sysfile,"#                   5       8     9       8    11\n");
 fprintf(sysfile,"# \n");
 fprintf(sysfile,"# The following formular is used:\n");
 fprintf(sysfile,"#                   X=(lX>>iDivShiftAx) + (lX>>iDivShiftBx);\n");
 fprintf(sysfile,"#                   Y=(lY>>iDivShiftAy) + (lY>>iDivShiftBy);\n");
 fprintf(sysfile,"# Note:\n");
 fprintf(sysfile,"# These settings can also be specified in the *.vol files,\n");
 fprintf(sysfile,"# which overwrite settings found here!\n");
 fprintf(sysfile,"DISPLAY_0SHIFTA_X               :%i\n", display_infox[0].iDivShiftA);
 fprintf(sysfile,"DISPLAY_0SHIFTB_X               :%i\n", display_infox[0].iDivShiftB);
 fprintf(sysfile,"DISPLAY_0SHIFTA_Y               :%i\n", display_infoy[0].iDivShiftA);
 fprintf(sysfile,"DISPLAY_0SHIFTB_Y               :%i\n", display_infoy[0].iDivShiftB);
 fprintf(sysfile,"DISPLAY_1SHIFTA_X               :%i\n", display_infox[1].iDivShiftA);
 fprintf(sysfile,"DISPLAY_1SHIFTB_X               :%i\n", display_infox[1].iDivShiftB);
 fprintf(sysfile,"DISPLAY_1SHIFTA_Y               :%i\n", display_infoy[1].iDivShiftA);
 fprintf(sysfile,"DISPLAY_1SHIFTB_Y               :%i\n", display_infoy[1].iDivShiftB);
 fprintf(sysfile,"DISPLAY_2SHIFTA_X               :%i\n", display_infox[2].iDivShiftA);
 fprintf(sysfile,"DISPLAY_2SHIFTB_X               :%i\n", display_infox[2].iDivShiftB);
 fprintf(sysfile,"DISPLAY_2SHIFTA_Y               :%i\n", display_infoy[2].iDivShiftA);
 fprintf(sysfile,"DISPLAY_2SHIFTB_Y               :%i\n", display_infoy[2].iDivShiftB);
 fprintf(sysfile,"DISPLAY_3SHIFTA_X               :%i\n", display_infox[3].iDivShiftA);
 fprintf(sysfile,"DISPLAY_3SHIFTB_X               :%i\n", display_infox[3].iDivShiftB);
 fprintf(sysfile,"DISPLAY_3SHIFTA_Y               :%i\n", display_infoy[3].iDivShiftA);
 fprintf(sysfile,"DISPLAY_3SHIFTB_Y               :%i\n", display_infoy[3].iDivShiftB);
 fprintf(sysfile,"DISPLAY_4SHIFTA_X               :%i\n", display_infox[4].iDivShiftA);
 fprintf(sysfile,"DISPLAY_4SHIFTB_X               :%i\n", display_infox[4].iDivShiftB);
 fprintf(sysfile,"DISPLAY_4SHIFTA_Y               :%i\n", display_infoy[4].iDivShiftA);
 fprintf(sysfile,"DISPLAY_4SHIFTB_Y               :%i\n", display_infoy[4].iDivShiftB);
 fprintf(sysfile,"DISPLAY_5SHIFTA_X               :%i\n", display_infox[5].iDivShiftA);
 fprintf(sysfile,"DISPLAY_5SHIFTB_X               :%i\n", display_infox[5].iDivShiftB);
 fprintf(sysfile,"DISPLAY_5SHIFTA_Y               :%i\n", display_infoy[5].iDivShiftA);
 fprintf(sysfile,"DISPLAY_5SHIFTB_Y               :%i\n", display_infoy[5].iDivShiftB);

 fprintf(sysfile,"\n# DISPLAY_VERBOSE (integer), values 0 or 1\n");
 fprintf(sysfile,"# When enabled some information is delivered to stdout\n");
 fprintf(sysfile,"# the screen might become garbled if this is enabled, so\n");
 fprintf(sysfile,"# if you do enable it, it would be wise to redirect the output\n");
 fprintf(sysfile,"# to some file.\n");
 fprintf(sysfile,"DISPLAY_VERBOSE                 :%i\n", bGblDisplayVerbose);

 fprintf(sysfile,"\n# MAX_SPEED (signed long integer), values 0 or 1000 (?)\n");
 fprintf(sysfile,"# This is a procentage value of the speed of a real\n");
 fprintf(sysfile,"# vectrex. So 100 means 100%% of a real vectrex.\n");
 fprintf(sysfile,"# On my AMD K6, 233 Mhz with Matrox Millenium I have been\n");
 fprintf(sysfile,"# able to get as high as 300%% of original speed, so a \n");
 fprintf(sysfile,"# delimiter is by now a real MUST.\n");
 fprintf(sysfile,"# Note:\n");
 fprintf(sysfile,"# The delimiter code is outside the processor emulation.\n");
 fprintf(sysfile,"# If FORCE_SINGLE_PROCESSOR is disabled the 6809 emulation is\n");
 fprintf(sysfile,"# run in little bursts of uninterruptable sequences.\n");
 fprintf(sysfile,"# That means as long as there is no interrupt expected the\n");
 fprintf(sysfile,"# 6809 emulation is taking over everything.\n");
 fprintf(sysfile,"# In the usual vectrex game this doesn't make a difference. \n");
 fprintf(sysfile,"# Test programs on the other hands, which do not assert \n");
 fprintf(sysfile,"# interrupts (or the like) will not be correctly slowed down.\n");
 fprintf(sysfile,"# Most noteably this appeared in Clay's SOUND.BIN, which is a\n");
 fprintf(sysfile,"# test program which plays sampled sound without any screen display\n");
 fprintf(sysfile,"# or input. It just bursts like hell (where speed is concerned) and\n");
 fprintf(sysfile,"# the sound is speeded up to. (sounds like it breathed in some helium :-))\n");
 fprintf(sysfile,"# Note 2:\n");
 fprintf(sysfile,"# If sound is played to fast inspite of 100%% speed, try setting \n");
 fprintf(sysfile,"# SOUND to 2, since the PSG emulation attempts to adjust to sampling\n");
 fprintf(sysfile,"# rates in relation to the speed of the vectrex.\n");
 fprintf(sysfile,"# (NOT done with SOUND set to 1).\n");
 fprintf(sysfile ,"MAX_SPEED                       :%i\n", savspeed);

 fprintf(sysfile,"\n# SAVE_ON_EXIT (integer), values 0 or 1\n");
 fprintf(sysfile,"# If enabled the vectrex state will be saved to a file.\n");
 fprintf(sysfile,"# The filename will be the name of the cartridge with \".sav\" as\n");
 fprintf(sysfile,"# extension. During emulation you can save to the same file with\n");
 fprintf(sysfile,"# \"ALT W\" and load with \"ALT L\".\n");
 fprintf(sysfile ,"SAVE_ON_EXIT                    :%i\n", bGblSaveOnExit);

 fprintf(sysfile,"\n# RESTORE_ON_START (integer), values 0 or 1\n");
 fprintf(sysfile,"# If enabled the emulator is looking for a file called\n");
 fprintf(sysfile,"# \"<name of the cartridge>.sav\", if found it will be\n");
 fprintf(sysfile,"# automatically loaded when emulation is started.\n");
 fprintf(sysfile ,"RESTORE_ON_START                :%i\n", bGblRestoreOnStart);

 fprintf(sysfile,"\n# DEBUG_ENABLE (integer), values 0 or 1\n");
 fprintf(sysfile,"# If enabled a debug window will be opened on start and\n");
 fprintf(sysfile,"# some messages will be displayed during emulation.\n");
 fprintf(sysfile,"# This is probably of little use to the day to day gamer.\n");
 fprintf(sysfile,"# For debugging the emulator and gleaming some knowledge on\n");
 fprintf(sysfile,"# how this thing works it is pretty usefull.\n");
 fprintf(sysfile,"# There are different DEBUG levels, they describe what kind of\n");
 fprintf(sysfile,"# information is actually displayed. You have to look at the sourcecode\n");
 fprintf(sysfile,"# for more information...\n");
 fprintf(sysfile,"# When DEBUG is active a completely different set of emulation\n");
 fprintf(sysfile,"# functions is working. On slower computer you will defenitly\n");
 fprintf(sysfile,"# want to disable DEBUG, for it is a slowdown garant.\n");
 fprintf(sysfile,"# This setting can be toggled online by ALT D.\n");
 fprintf(sysfile,"# NOTE:\n");
 fprintf(sysfile,"# If enabled GUI is enabled too!\n");
 fprintf(sysfile ,"DEBUG_ENABLE                    :%i\n", bGblDebugEnable);

 fprintf(sysfile,"\n# DEBUG_MONITOR (integer), values 0 or 1\n");
 fprintf(sysfile,"# If enabled a MONITOR window will be opened on start.\n");
 fprintf(sysfile,"# In this window you can do all the things you want to do\n");
 fprintf(sysfile,"# and are allowed to do. :-)\n");
 fprintf(sysfile,"# This setting can be toggled online by ALT M (or invoke the\n");
 fprintf(sysfile,"# exit function in the Monitor window...).\n");
 fprintf(sysfile,"# NOTE:\n");
 fprintf(sysfile,"# If enabled GUI is enabled too!\n");
 fprintf(sysfile ,"DEBUG_MONITOR                   :%i\n", bGblDebugMonitorEnable);

 fprintf(sysfile,"\n# DEBUG_MONITOR_PAUSE (integer), values 0 or 1\n");
 fprintf(sysfile,"# Should the emulator be paused on monitor startup?\n");
 fprintf(sysfile ,"DEBUG_MONITOR_PAUSE             :%i\n", pause_on_monitor_entry_default);

 fprintf(sysfile,"\n# FORCE_SINGLE_PROCESSOR (integer), values 0 or 1\n");
 fprintf(sysfile,"# There are different ways, how a real vectrex cycle is\n");
 fprintf(sysfile,"# emulated internally. I call them single and multiprocessor.\n");
 fprintf(sysfile,"# That has nothing to do with being capable of using more than\n");
 fprintf(sysfile,"# one processor. It originates in how the processor and the\n");
 fprintf(sysfile,"# custom chips are updated internally.\n");
 fprintf(sysfile,"# Multi means that the processor is running full speed,\n");
 fprintf(sysfile,"# no custom chips are updated, as long as no custom chip\n");
 fprintf(sysfile,"# information is needed.\n");
 fprintf(sysfile,"# Single means that the processor and custom chips are allways updated\n");
 fprintf(sysfile,"# every 'emulation round'. (which means it is bloody slow going)\n");
 fprintf(sysfile,"# Actually it shouldn't really be neccessary to activate this option,\n");
 fprintf(sysfile,"# since, if everything is done all right, the only visual effect is a \n");
 fprintf(sysfile,"# dramatic slowdown of everything. I thought at some stage it might\n");
 fprintf(sysfile,"# be neccessary, but I don't think it really is, but I left it in for\n");
 fprintf(sysfile,"# being to lazy to remove it. Perhaps, if you have a fast processor some\n");
 fprintf(sysfile,"# sounds might be cleaner...\n");
 fprintf(sysfile,"# Someday, when I get around doing it, I will use this setting to \n");
 fprintf(sysfile,"# get the speedthrottling working correctly...\n");
 fprintf(sysfile ,"FORCE_SINGLE_PROCESSOR          :%i\n", bGblForceSingleProcessor);

 fprintf(sysfile,"\n# GUI_ENABLE (integer), values 0 or 1\n");
 fprintf(sysfile,"# Oh well, it says it, doesn't it?\n");
 fprintf(sysfile,"# Gets the GUI running right from start.\n");
 fprintf(sysfile,"# You can toggle it using <SPACE> while running.\n");
 fprintf(sysfile,"# On slower systems you perhaps don't want the GUI to be\n");
 fprintf(sysfile,"# running, since it steals some time...\n");
 fprintf(sysfile ,"GUI_ENABLE                      :%i\n", bGblGUIEnable);

 fprintf(sysfile,"\n# SOUND (unsigned short integer), values 0 to 2\n");
 fprintf(sysfile,"# There are now three settings.\n");
 fprintf(sysfile,"# 0 - No sound at all\n");
 fprintf(sysfile,"# 1 - Marcel De Kogel's emulation of the AY-3-8912,\n");
 fprintf(sysfile,"#     Adlib (Soundblaster for digital) (the old way)\n");
 fprintf(sysfile,"# 2 - PSG emulation by Ville Hallik, Michael Cuddy and\n");
 fprintf(sysfile,"#     (some really minor changes for digital sound)\n");
 fprintf(sysfile,"#     Christopher Salomon,\n");
 fprintf(sysfile,"#     SEAL Audio library, thanks guys...\n");
 fprintf(sysfile ,"SOUND                           :%i\n", wGblSound);

 fprintf(sysfile,"\n# SEAL_AUDIO_DEVICE (unsigned short integer), values 0 to 65535\n");
 fprintf(sysfile,"# Only interesting if SOUND is set to 2.\n");
 fprintf(sysfile,"# Lets you force the SEAL library to use a certain audio\n");
 fprintf(sysfile,"# device. For information what devices are available\n");
 fprintf(sysfile,"# start the emulator with 'sealinfo' as a command line\n");
 fprintf(sysfile,"# parameter.\n");
 fprintf(sysfile,"# For automatic detection (recomended) use 65535.\n");
 fprintf(sysfile ,"SEAL_AUDIO_DEVICE               :%i\n", (T_UWORD)wGblseal_audio_device);

 fprintf(sysfile,"\n# DIGITAL_VOLUME (unsigned short integer), values 0 to 100 (or more?) \n");
 fprintf(sysfile,"# Only interesting if SOUND is set to 1.\n");
 fprintf(sysfile,"# This setting is used for changing the volume of digitzed sound\n");
 fprintf(sysfile,"# only! It applies to the chunks of sound given to the DAC of\n");
 fprintf(sysfile,"# the Soundblaster. Thus the (now) only available cartridge\n");
 fprintf(sysfile,"# effected is 'SPIKE'.\n");
 fprintf(sysfile,"# This setting was more built for my personal use, since\n");
 fprintf(sysfile,"# I needed some method to compensate the different settings\n");
 fprintf(sysfile,"# for ADLIB volume and DAC output volume.\n");
 fprintf(sysfile,"# If you find your digital output to loud or to soft,\n");
 fprintf(sysfile,"# you might experiment with this value, it gives\n");
 fprintf(sysfile,"# a percentage of the original volume to output.\n");
 fprintf(sysfile,"# So for really soft output use small values.\n");
 fprintf(sysfile ,"DIGITAL_VOLUME                  :%i\n", wGblDigitalVolume);

 fprintf(sysfile,"\n# ROM_WRITE_ENABLE (integer), values 0 to 1\n");
 fprintf(sysfile,"# Well, just what it says.\n");
 fprintf(sysfile,"# Actually I don't really know why Keith put it in here,\n");
 fprintf(sysfile,"# I guess it is usefull, but what for? (I don't know)\n");
 fprintf(sysfile ,"ROM_WRITE_ENABLE                :%i\n", bGblSystemROMWriteEnable);

 fprintf(sysfile,"\n# CART_WRITE_ENABLE (integer), values 0 to 1\n");
 fprintf(sysfile,"# Same as above.\n");
 fprintf(sysfile,"# Well, just what it says.\n");
 fprintf(sysfile,"# Actually I don't really know why Keith put it in here,\n");
 fprintf(sysfile,"# I guess it is usefull, but what for? (I don't know)\n");
 fprintf(sysfile ,"CART_WRITE_ENABLE               :%i\n", bGblCartridgeROMWriteEnable);

 fprintf(sysfile,"\n# PCJOYSTICK (unsigned short integer), values 0 to 2\n");
 fprintf(sysfile,"# Well, joystick support certainly took its time. \n");
 fprintf(sysfile,"# 0 - no joystick is used (a bit of a speed up)\n");
 fprintf(sysfile,"# 1 - PC joystick is used, but input is taken as\n");
 fprintf(sysfile,"#     digital input (even with an analog joystick)\n");
 fprintf(sysfile,"# 2 - PC joystick is used and input is used as\n");
 fprintf(sysfile,"#     analog input\n");
 fprintf(sysfile,"# The calibration is done automatically, just wiggle it\n");
 fprintf(sysfile,"# a few times at startup, to give the code a chance to get a \n");
 fprintf(sysfile,"# grip on things.\n");
 fprintf(sysfile,"# Polling the damn PC joystick allways takes a lot of time,\n");
 fprintf(sysfile,"# on slow systems you perhaps want to disable this to\n");
 fprintf(sysfile,"# use the keyboard only...\n");
 fprintf(sysfile,"# NOTE:\n");
 fprintf(sysfile,"# Keyboard (joystick) controls are disabled when\n");
 fprintf(sysfile,"# Joystick is active.\n");
 fprintf(sysfile,"# Button 4 is mapped to Joystick Button A\n");
 fprintf(sysfile,"# Button 3 is mapped to Joystick Button B\n");
 fprintf(sysfile,"# Button 1,2,3,4 are also all still mapped to the keyboard.\n");
 fprintf(sysfile ,"PCJOYSTICK                      :%i\n", wGblPCJoystick);

 fprintf(sysfile,"\n# DIGITAL_JOYSTICK (integer), values 0 to 1\n");
 fprintf(sysfile,"# This is only used when the above PCJOYSTICK is set to 0.\n");
 fprintf(sysfile,"# If (DIGITAL_JOYSTICK is) set to 1, the input from joystick\n");
 fprintf(sysfile,"# keys is send to the vectrex as the maximum possible analog\n");
 fprintf(sysfile,"# joystick setting of the vectrex. Many cartridges do not use the\n");
 fprintf(sysfile,"# analog information the original vectrex joystick was capable of sending.\n");
 fprintf(sysfile,"# It is (sort of) safer for these games to use this setting.\n");
 fprintf(sysfile,"# And perhaps a tiny little bit faster (but don't be concerned).\n");
 fprintf(sysfile,"# If set to analog (0), the direction keys are (sort of) polled,\n");
 fprintf(sysfile,"# and the longer you hold the key, the more 'direction' is emulated.\n");
 fprintf(sysfile ,"DIGITAL_JOYSTICK                :%i\n", bGblDigitalJoystick);

 fprintf(sysfile,"\n# JOYSTICK_AUTO_CENTRE (integer), values 0 to 1\n");
 fprintf(sysfile,"# This is only used when the above PCJOYSTICK is set to 0 and\n");
 fprintf(sysfile,"# DIGITAL_JOYSTICK is set to 0. That means the keyboard direction keys\n");
 fprintf(sysfile,"# are used to emulate an analog joystick.\n");
 fprintf(sysfile,"# If autocenter is enabled, the above mentioned increased 'direction'\n");
 fprintf(sysfile,"# (increased by holding the keys down for a certain period of time)\n");
 fprintf(sysfile,"# is decreased back to center, when no direction key is pressed.\n");
 fprintf(sysfile,"# Otherwise, the 'direction' stays the same, till some other direction key\n");
 fprintf(sysfile,"# is pressed.\n");
 fprintf(sysfile ,"JOYSTICK_AUTO_CENTRE            :%i\n", bGblJoystickAutoCentre);

 fprintf(sysfile,"\n# JOYSTICK_SENSITIVITY (signed short integer), values 0 to 32767\n");
 fprintf(sysfile,"# This is only used when the above PCJOYSTICK is set to 0 and\n");
 fprintf(sysfile,"# DIGITAL_JOYSTICK is set to 0. That means the keyboard direction keys\n");
 fprintf(sysfile,"# are used to emulate an analog joystick.\n");
 fprintf(sysfile,"# This is the value, which is added (subtracted) upon every poll of\n");
 fprintf(sysfile,"# the direction key. The maximum 'direction' possible is +-32767.\n");
 fprintf(sysfile,"# A nice value is for example 16.\n");
 fprintf(sysfile ,"JOYSTICK_SENSITIVITY            :%i\n", (T_INT)((T_SWORD)wGblJoystickSensitivity));

 fprintf(sysfile,"\n# PLAYER#$$$$ (unsigned short integer), values 0 to 32767\n");
 fprintf(sysfile,"# Well these define the keys which are used for each player.\n");
 fprintf(sysfile,"# It would be to much, to list all keys here, for a comprehensive\n");
 fprintf(sysfile,"# list, look at some book, or in the header file 'scandef.h'.\n");
 fprintf(sysfile,"# Some codes:\n");
 fprintf(sysfile,"# (taken from the ZSNES doc, thanks guys)\n");
 fprintf(sysfile,"#  1  ESC         21 Y        41  ~        61  F3       81  PGDN\n");
 fprintf(sysfile,"#  2  1           22 U        42  LSHFT    62  F4       82  INS\n");
 fprintf(sysfile,"#  3  2           23 I        43  \\        63  F5       83  DEL\n");
 fprintf(sysfile,"#  4  3           24 O        44  Z        64  F6\n");
 fprintf(sysfile,"#  5  4           25 P        45  X        65  F7\n");
 fprintf(sysfile,"#  6  5           26 [        46  C        66  F8\n");
 fprintf(sysfile,"#  7  6           27 ]        47  V        67  F9\n");
 fprintf(sysfile,"#  8  7           28 ENTER    48  B        68  F10\n");
 fprintf(sysfile,"#  9  8           29 CTRL     49  N        69  NUM\n");
 fprintf(sysfile,"# 10  9           30 A        50  M        70  SCRL\n");
 fprintf(sysfile,"# 11  0           31 S        51  ,        71  HOME\n");
 fprintf(sysfile,"# 12  -           32 D        52  .        72  UP\n");
 fprintf(sysfile,"# 13  =           33 F        53  /        73  PGUP\n");
 fprintf(sysfile,"# 14  BACKSPC     34 G        54  RSHFT    74  -\n");
 fprintf(sysfile,"# 15  TAB         35 H        55  PRTSC    75  LEFT\n");
 fprintf(sysfile,"# 16  Q           36 J        56  ALT      76  KEY5\n");
 fprintf(sysfile,"# 17  W           37 K        57  SPACE    77  RIGHT\n");
 fprintf(sysfile,"# 18  E           38 L        58  CAPS     78\n");
 fprintf(sysfile,"# 19  R           39 ;        59  F1       79  END\n");
 fprintf(sysfile,"# 20  T           40 '        60  F2       80  DOWN\n");
 fprintf(sysfile ,"PLAYER1BUTTON1                  :%i\n", wGblPly1Button1);
 fprintf(sysfile ,"PLAYER1BUTTON2                  :%i\n", wGblPly1Button2);
 fprintf(sysfile ,"PLAYER1BUTTON3                  :%i\n", wGblPly1Button3);
 fprintf(sysfile ,"PLAYER1BUTTON4                  :%i\n", wGblPly1Button4);
 fprintf(sysfile ,"PLAYER1UP                       :%i\n", wGblPly1JoystkUp);
 fprintf(sysfile ,"PLAYER1DOWN                     :%i\n", wGblPly1JoystkDn);
 fprintf(sysfile ,"PLAYER1LEFT                     :%i\n", wGblPly1JoystkLf);
 fprintf(sysfile ,"PLAYER1RIGHT                    :%i\n", wGblPly1JoystkRt);
 fprintf(sysfile ,"PLAYER2BUTTON1                  :%i\n", wGblPly2Button1);
 fprintf(sysfile ,"PLAYER2BUTTON2                  :%i\n", wGblPly2Button2);
 fprintf(sysfile ,"PLAYER2BUTTON3                  :%i\n", wGblPly2Button3);
 fprintf(sysfile ,"PLAYER2BUTTON4                  :%i\n", wGblPly2Button4);
 fprintf(sysfile ,"PLAYER2UP                       :%i\n", wGblPly2JoystkUp);
 fprintf(sysfile ,"PLAYER2DOWN                     :%i\n", wGblPly2JoystkDn);
 fprintf(sysfile ,"PLAYER2LEFT                     :%i\n", wGblPly2JoystkLf);
 fprintf(sysfile ,"PLAYER2RIGHT                    :%i\n", wGblPly2JoystkRt);

 fprintf(sysfile,"\n# TICKSTOGRAB (unsigned long integer), values 0 to <very high number>\n");
 fprintf(sysfile,"# Don't worry about this.\n");
 fprintf(sysfile,"# It is not compiled into this version.\n");
 fprintf(sysfile,"# But for the curious, it defines the 'ticks' between\n");
 fprintf(sysfile,"# two screenshots. Clay wanted to do a small video/animation\n");
 fprintf(sysfile,"# and asked me to implement a continous snapshot feature.\n");
 fprintf(sysfile,"# I'm not sure, if the code still works, haven't compiled it\n");
 fprintf(sysfile,"# in ages...\n");
 fprintf(sysfile ,"TICKSTOGRAB                     :%i\n", lwGblTicksToGrab);

 fprintf(sysfile,"\n# BINARY_DIRECTORY (string), values, well a string!\n");
 fprintf(sysfile,"# The default directory, where the binary rom images\n");
 fprintf(sysfile,"# are looked for first.\n");
 fprintf(sysfile,"# The path can be either absolute or relative.\n");
 fprintf(sysfile ,"BINARY_DIRECTORY                :\"");
 i=0;
 while (*(GblBinDir+i)!=0)
 {
  fprintf(sysfile ,"%c",*(GblBinDir+i));
  if (*(GblBinDir+i)=='\\')
   fprintf(sysfile ,"\\");
  i++;
 }
 fprintf(sysfile ,"\"\n");

 fprintf(sysfile,"\n# VOL_FILE_DIRECTORY (string), values, well a string!\n");
 fprintf(sysfile,"# The default directory, where the *.vol files\n");
 fprintf(sysfile,"# are looked for first.\n");
 fprintf(sysfile,"# The path can be either absolute or relative.\n");
 fprintf(sysfile ,"VOL_FILE_DIRECTORY              :\"");
 i=0;
 while (*(GblVolDir+i)!=0)
 {
  fprintf(sysfile ,"%c",*(GblVolDir+i));
  if (*(GblVolDir+i)=='\\')
   fprintf(sysfile ,"\\");
  i++;
 }
 fprintf(sysfile ,"\"\n");

 fprintf(sysfile,"\n# PICTURE_DIRECTORY (string), values, well a string!\n");
 fprintf(sysfile,"# The default directory, where the *.pcx files\n");
 fprintf(sysfile,"# are looked for first.\n");
 fprintf(sysfile,"# The path can be either absolute or relative.\n");
 fprintf(sysfile ,"PICTURE_DIRECTORY               :\"");
 i=0;
 while (*(GblPicDir+i)!=0)
 {
  fprintf(sysfile ,"%c",*(GblPicDir+i));
  if (*(GblPicDir+i)=='\\')
   fprintf(sysfile ,"\\");
  i++;
 }
 fprintf(sysfile ,"\"\n");

 fprintf(sysfile,"\n# SAVE_DIRECTORY (string), values, well a string!\n");
 fprintf(sysfile,"# The default directory, where the save files\n");
 fprintf(sysfile,"# are looked for first.\n");
 fprintf(sysfile,"# The path can be either absolute or relative.\n");
 fprintf(sysfile ,"SAVE_DIRECTORY                  :\"");
 i=0;
 while (*(GblSvgDir+i)!=0)
 {
  fprintf(sysfile ,"%c",*(GblSvgDir+i));
  if (*(GblSvgDir+i)=='\\')
   fprintf(sysfile ,"\\");
  i++;
 }
 fprintf(sysfile ,"\"\n");

 fprintf(sysfile,"\n# DEV_DIRECTORY (string), values, well a string!\n");
 fprintf(sysfile,"# The default directory, where the development files\n");
 fprintf(sysfile,"# are looked for first.\n");
 fprintf(sysfile,"# The path can be either absolute or relative.\n");
 fprintf(sysfile ,"DEV_DIRECTORY                  :\"");
 i=0;
 while (*(GblDevDir+i)!=0)
 {
  fprintf(sysfile ,"%c",*(GblDevDir+i));
  if (*(GblDevDir+i)=='\\')
   fprintf(sysfile ,"\\");
  i++;
 }
 fprintf(sysfile ,"\"\n");

 fprintf(sysfile,"\n# ROM_IMAGE (string), values, well a string!\n");
 fprintf(sysfile,"# Well, the default 'operating system' to load.\n");
 fprintf(sysfile ,"ROM_IMAGE                       :\"%s\"\n", acGblROMImageFileName);

 fprintf(sysfile,"\n# RAM_IMAGE (string), values, well a string!\n");
 fprintf(sysfile,"# Actually, I dunno, really.\n");
 fprintf(sysfile,"# It should specify a ram image (1KB), which might be loaded\n");
 fprintf(sysfile,"# upon startup, but there is no feature to save one :-).\n");
 fprintf(sysfile,"# I guess Keith wanted to implement a game/snapshot with this\n");
 fprintf(sysfile,"# or some kind of debugging? Anyway, it is still in here, and\n");
 fprintf(sysfile,"# fairly useless. (anyone tried to enter a name here?)\n");
 fprintf(sysfile ,"RAM_IMAGE                       :\"%s\"\n", acGblRAMImageFileName);

 fprintf(sysfile,"\n# DEFAULT_CART (string), values, well a string!\n");
 fprintf(sysfile,"# I wonder what this might be for...\n");
 fprintf(sysfile ,"DEFAULT_CART                    :\"%s\"\n", acGblDefaultCartFileName);

 fprintf(sysfile,"\n# CARTRIDGE## (string), values, well a string!\n");
 fprintf(sysfile,"# The carousel cartridge names, for ALT F1 to ALT F12...!\n");
 fprintf(sysfile ,"CARTRIDGE01                     :\"%s\"\n", acGblCartCarousel[0]);
 fprintf(sysfile ,"CARTRIDGE02                     :\"%s\"\n", acGblCartCarousel[1]);
 fprintf(sysfile ,"CARTRIDGE03                     :\"%s\"\n", acGblCartCarousel[2]);
 fprintf(sysfile ,"CARTRIDGE04                     :\"%s\"\n", acGblCartCarousel[3]);
 fprintf(sysfile ,"CARTRIDGE05                     :\"%s\"\n", acGblCartCarousel[4]);
 fprintf(sysfile ,"CARTRIDGE06                     :\"%s\"\n", acGblCartCarousel[5]);
 fprintf(sysfile ,"CARTRIDGE07                     :\"%s\"\n", acGblCartCarousel[6]);
 fprintf(sysfile ,"CARTRIDGE08                     :\"%s\"\n", acGblCartCarousel[7]);
 fprintf(sysfile ,"CARTRIDGE09                     :\"%s\"\n", acGblCartCarousel[8]);
 fprintf(sysfile ,"CARTRIDGE10                     :\"%s\"\n", acGblCartCarousel[9]);
 fprintf(sysfile ,"CARTRIDGE11                     :\"%s\"\n", acGblCartCarousel[10]);
 fprintf(sysfile ,"CARTRIDGE12                     :\"%s\"\n", acGblCartCarousel[11]);
#ifdef ANALOG_CYCLE_PATCH

 fprintf(sysfile ,"\n\n");
 fprintf(sysfile ,"BLANK                           :%i\n", BLANK_REACT_OFFSET_CYCLES);
 fprintf(sysfile ,"RAMP                            :%i\n", RAMP_REACT_OFFSET_CYCLES);
 fprintf(sysfile ,"SAMPLE_HOLD                     :%i\n", SAMPLE_HOLD_REACT_OFFSET_CYCLES);
 fprintf(sysfile ,"DAC                             :%i\n", DAC_REACT_OFFSET_CYCLES);
 fprintf(sysfile ,"ZERO                            :%i\n", ZERO_REACT_OFFSET_CYCLES);
#endif
 fclose (sysfile);
 return T_OK;
}

T_INT execute_vhelp_button(PARAMETER *parameter)
{
 parameter->executed=PARAMETER_EXECUTED;
 if (fKBDisALT()==T_TRUE)
 {
  help_function("thelp.hlf",NULL);
 }
 else
  help_function("index.hlf",NULL);
 return parameter->ebene;
}

T_INT execute_vgame_button(PARAMETER *parameter)
{
 parameter->executed=PARAMETER_EXECUTED;
 if (fKBDisALT()==T_TRUE)
 {
  help_function("tgames.hlf",NULL);
 }
 else
  help_function("vgames.hlf",NULL);
 return parameter->ebene;
}

T_INT execute_vload_button(PARAMETER *parameter)
{
 static WINDOW_LISTEN_ELEMENT *window=NULL;
 OPEN_WIN w;
 BUTTON *button=NULL;
 parameter->executed=PARAMETER_EXECUTED;
 if (fKBDisALT()==T_TRUE)
 {
  help_function("tload.hlf",NULL);
  return parameter->ebene;
 }
 if (window==NULL)
 {
  T_CHAR button_name[2];
  T_INT i;
  button_name[0]='0';
  button_name[1]=(T_CHAR)0;
  w.x_pos =vesa_x-(10*(8+4)+5); // 2*2 is border +1 is button offset
  w.y_pos =(default_button_size+2+2*2+2*2); // is 2* linewidth of button
  w.breite=vesa_x-w.x_pos;
  w.hoehe=(16+4+1);
  w.border=2;
  w.color=ddark_grau;
  w.bcolor=dhell_grau;
  w.art=RAW_WINDOW|REMEMBER;
  w.id=WIN_OPEN_CHOOSES_ID;
  w.variablen=NULL;
  w.mem_bild=NULL;
  w.pcx_bild=NULL;
  w.name=NULL;
  w.destructor=NULL;
  w.area_funktion=execute_win_move;
  w.tasten_funktion=NULL;
  window=open_window(&w);
  if (window==NULL)
  {
   t_printf(0,"Error config Window!");
   return parameter->ebene;
  }
  print_srect(window,PRINT_WIN_RAW,0,0,window->b,window->h,ddark_grau,weiss,2);
  for (i=0;i<10;i++)
  {
   button = make_small_text_button(button_name,hhell_grau,hdark_grau,ddark_grau,mdark_grau,dhell_grau,mhell_grau,2,KEY_C,SWITCH_ON_RELEASED);
   if (button==NULL)
   {
    t_printf(0,"Error while initialising load buttons");
    return parameter->ebene;
   }
   add_wbutton(window,button,0+i*(8+4),0,0);
   button_name[0]++;
   button->button_funktion=execute_vload_game_button;
  }
 } // if (window==NULL)
 else
 {
  close_window(window,CLOSE_DEFAULT);
  window=NULL;
 }
 return parameter->ebene;
}

T_INT start_game_file(T_PCHAR filename)
{
 strcpy(acGblDefaultCartFileName,filename);
 init_after_execution=T_TRUE;
 action_after_execution=V_LOAD_CARTRIDGE;
 return T_OK;
}

T_INT execute_vload_game_button(PARAMETER *parameter)
{
 parameter->executed=PARAMETER_EXECUTED;
 if (fKBDisALT()==T_TRUE)
 {
  help_function("tlgame.hlf",NULL);
 }
 else
 {
  if (parameter->button_element!=NULL)
  {
   BUTTON_LISTEN_ELEMENT *button=parameter->button_element;
   init_after_execution=T_TRUE;
   action_after_execution=V_LOAD_SAVED_GAME;
   load_var=button->button->name[0]-'0'+KEY_0;
  }
 }
 return parameter->ebene;
}

T_INT execute_vexit_button(PARAMETER *parameter)
{
 parameter->executed=PARAMETER_EXECUTED;
 if (fKBDisALT()==T_TRUE)
 {
  help_function("texit.hlf",NULL);
 }
 else
 {
  init_after_execution=T_TRUE;
  action_after_execution=V_QUIT;
 }
 return parameter->ebene;
}

T_INT execute_vreboot_button(PARAMETER *parameter)
{
 parameter->executed=PARAMETER_EXECUTED;
 if (fKBDisALT()==T_TRUE)
 {
  help_function("treboot.hlf",NULL);
 }
 else
 {
  init_after_execution=T_TRUE;
  action_after_execution=V_REBOOT;
 }
 return parameter->ebene;
}

T_VOID init_vectrex_gui(T_VOID)
{
 WINDOW_LISTEN_ELEMENT *window=NULL;
 OPEN_WIN w;
 BUTTON *button=NULL;
 w.x_pos =vesa_x-(default_button_size+2+2*2+2*2); // 2*2 is border +2 is button offset
 w.y_pos =vesa_y-(default_button_size+2+2*2+2*2); // is 2* linewidth of button
 w.breite=default_button_size+2+2*2;
 w.hoehe=default_button_size+2+2*2;
 w.border=2;
 w.color=ddark_grau;
 w.bcolor=dhell_grau;
 w.art=RAW_WINDOW|BACK_WINDOW|REMEMBER;
 w.id=WIN_OPEN_CHOOSES_ID;
 w.variablen=NULL;
 w.mem_bild=NULL;
 w.pcx_bild=NULL;
 w.name=NULL;
 w.destructor=NULL;
 w.area_funktion=execute_win_move;
 w.tasten_funktion=NULL;
 window=open_window(&w);
 if (window==NULL)
 {
  t_printf(0,"Error while initialising GUI");
  return;
 }
 print_srect(window,PRINT_WIN_RAW,0,0,window->b,window->h,ddark_grau,weiss,2);
 button = make_mem_bild_button(exit_pic,default_back_pixel,ddark_grau,mdark_grau,dhell_grau,mhell_grau,2,KEY_ALT_X,SWITCH_ON_RELEASED);
 if (button==NULL)
 {
  t_printf(0,"Error while initialising GUI");
  return;
 }
 add_wbutton(window,button,1,1,0);
 button->button_funktion=execute_vexit_button;
 w.x_pos =vesa_x-(default_button_size+2+2*2+2*2); // 2*2 is border +2 is button offset
 w.y_pos =0;
 window=open_window(&w);
 if (window==NULL)
 {
  t_printf(0,"Error while initialising GUI");
  return;
 }
 print_srect(window,PRINT_WIN_RAW,0,0,window->b,window->h,ddark_grau,weiss,2);
 button = make_mem_bild_button(help_pic,default_back_pixel,ddark_grau,mdark_grau,dhell_grau,mhell_grau,2,KEY_F1,SWITCH_ON_RELEASED);
 if (button==NULL)
 {
  t_printf(0,"Error while initialising GUI");
  return;
 }
 add_wbutton(window,button,1,1,0);
 button->button_funktion=execute_vhelp_button;

 w.x_pos =vesa_x-(default_button_size+2+2*2+2*2)-(default_button_size+2+2*2+2*2); // 2*2 is border +2 is button offset
 w.y_pos =0;
 window=open_window(&w);
 if (window==NULL)
 {
  t_printf(0,"Error while initialising GUI");
  return;
 }
 print_srect(window,PRINT_WIN_RAW,0,0,window->b,window->h,ddark_grau,weiss,2);
 button = make_mem_bild_button(game_pic,default_back_pixel,ddark_grau,mdark_grau,dhell_grau,mhell_grau,2,KEY_G,SWITCH_ON_RELEASED);
 if (button==NULL)
 {
  t_printf(0,"Error while initialising GUI");
  return;
 }
 add_wbutton(window,button,1,1,0);
 button->button_funktion=execute_vgame_button;

 w.x_pos =vesa_x-3*(default_button_size+2+2*2+2*2);
 w.y_pos =0;
 window=open_window(&w);
 if (window==NULL)
 {
  t_printf(0,"Error while initialising GUI");
  return;
 }
 print_srect(window,PRINT_WIN_RAW,0,0,window->b,window->h,ddark_grau,weiss,2);
 button = make_mem_bild_button(config_pic,default_back_pixel,ddark_grau,mdark_grau,dhell_grau,mhell_grau,2,KEY_C,SWITCH_ON_RELEASED);
 if (button==NULL)
 {
  t_printf(0,"Error while initialising GUI");
  return;
 }
 add_wbutton(window,button,1,1,0);
 button->button_funktion=execute_vconfig_button;

 w.x_pos =vesa_x-4*(default_button_size+2+2*2+2*2);
 w.y_pos =0;
 window=open_window(&w);
 if (window==NULL)
 {
  t_printf(0,"Error while initialising GUI");
  return;
 }
 print_srect(window,PRINT_WIN_RAW,0,0,window->b,window->h,ddark_grau,weiss,2);
 button = make_mem_bild_button(read_pic,default_back_pixel,ddark_grau,mdark_grau,dhell_grau,mhell_grau,2,KEY_S,SWITCH_ON_RELEASED);
 if (button==NULL)
 {
  t_printf(0,"Error while initialising GUI");
  return;
 }
 add_wbutton(window,button,1,1,0);
 button->button_funktion=execute_vload_button;

 w.x_pos =vesa_x-5*(default_button_size+2+2*2+2*2);
 w.y_pos =0;
 window=open_window(&w);
 if (window==NULL)
 {
  t_printf(0,"Error while initialising GUI");
  return;
 }
 print_srect(window,PRINT_WIN_RAW,0,0,window->b,window->h,ddark_grau,weiss,2);
 button = make_mem_bild_button(screen_pic,default_back_pixel,ddark_grau,mdark_grau,dhell_grau,mhell_grau,2,KEY_SPACE,SWITCH_ON_RELEASED);
 if (button==NULL)
 {
  t_printf(0,"Error while initialising GUI");
  return;
 }
 add_wbutton(window,button,1,1,0);
 button->button_funktion=execute_vtoggle_gui_button;

 w.x_pos =vesa_x-6*(default_button_size+2+2*2+2*2);
 w.y_pos =0;
 window=open_window(&w);
 if (window==NULL)
 {
  t_printf(0,"Error while initialising GUI");
  return;
 }
 print_srect(window,PRINT_WIN_RAW,0,0,window->b,window->h,ddark_grau,weiss,2);
 button = make_mem_bild_button(reboot_pic,default_back_pixel,ddark_grau,mdark_grau,dhell_grau,mhell_grau,2,0,SWITCH_ON_RELEASED);
 if (button==NULL)
 {
  t_printf(0,"Error while initialising GUI");
  return;
 }
 add_wbutton(window,button,1,1,0);
 button->button_funktion=execute_vreboot_button;
}

T_INT execute_vconfig_button(PARAMETER *parameter)
{
 static WINDOW_LISTEN_ELEMENT *window=NULL;
 OPEN_WIN w;
 BUTTON *button=NULL;
 parameter->executed=PARAMETER_EXECUTED;
 if (fKBDisALT()==T_TRUE)
 {
  help_function("tconfig.hlf",NULL);
  return parameter->ebene;
 }
 if (window==NULL)
 {
  w.breite=200;
  w.hoehe=100;
  w.x_pos =vesa_x/2-w.breite/2;
  w.y_pos =0;
  w.border=2;
  w.color=ddark_grau;
  w.bcolor=dhell_grau;
  w.art=RAW_WINDOW|REMEMBER;
  w.id=WIN_OPEN_CHOOSES_ID;
  w.variablen=NULL;
  w.mem_bild=NULL;
  w.pcx_bild=NULL;
  w.name=NULL;
  w.destructor=NULL;
  w.area_funktion=execute_win_move;
  w.tasten_funktion=NULL;
  window=open_window(&w);
  if (window==NULL)
  {
   t_printf(0,"Error config Window!");
   return parameter->ebene;
  }
  print_srect(window,PRINT_WIN_RAW,0,0,window->b,window->h,ddark_grau,weiss,2);
  button = make_small_text_button("Edit \"vectrex.ini\"",hhell_grau,hdark_grau,ddark_grau,mdark_grau,dhell_grau,mhell_grau,2,KEY_C,SWITCH_ON_RELEASED);
  if (button==NULL)
  {
   t_printf(0,"Error while initialising load buttons");
   return parameter->ebene;
  }
  add_wbutton(window,button,1,1,0);
  button->button_funktion=execute_vedit_ini_button;
 } // if (window==NULL)
 else
 {
  close_window(window,CLOSE_DEFAULT);
  window=NULL;
 }
 return parameter->ebene;
}

T_INT execute_vedit_ini_button(PARAMETER *parameter)
{
 OPEN_WIN w;
 WINDOW_LISTEN_ELEMENT *text_window;
 T_INT breite,hoehe;
 T_INT x=100,y=100;
 T_INT id=WIN_OPEN_CHOOSES_ID;
 parameter->executed=PARAMETER_EXECUTED;
 if (fKBDisALT()==T_TRUE)
 {
  help_function("teditini.hlf",NULL);
  return parameter->ebene;
 }
 breite=80*8;
 hoehe=25*16;
 x=vesa_x/2-(breite/2);
 y=vesa_y/2-(hoehe/2);
 w.x_pos=x;
 w.y_pos=y;
 w.breite=breite;
 w.hoehe=hoehe;
 w.border=1;
 w.bcolor=dhell_grau;
 w.color=ddark_grau;
 w.art=RAW_WINDOW|WITH_CLOSE_BUTTON|WITH_HIDE_BUTTON|WITH_DRAG_BUTTON|VIRTUAL_FRONT_WINDOW|REMEMBER|EDIT_TRUE;
 w.id=id;
 w.name=NULL;
 w.variablen=NULL;
 w.mem_bild=NULL;
 w.pcx_bild=NULL;
 w.destructor=NULL;
 w.area_funktion=execute_activate;
 w.tasten_funktion=NULL;
 text_window=open_window(&w);
 if (text_window==NULL)
 {
  return parameter->ebene;
 }
 is_upper=T_TRUE;
 print_srect(text_window,PRINT_WIN_RAW,0,0,text_window->b,text_window->h,ddark_grau,weiss,1);
 is_upper=T_FALSE;
 edit(text_window,"vectrex.ini",EDIT_EDIT|EDIT_START|EDIT_WITH_LOAD|EDIT_WITH_SAVE);
 return parameter->ebene;
}

T_INT execute_vtoggle_gui_button(PARAMETER *parameter)
{
 if (fKBDisALT()==T_TRUE)
 {
  help_function("tgui.hlf",NULL);
 }
 else
 {
  if (bGblPauseMode==T_FALSE)
  {
   bGblGUIEnable=T_FALSE;
   hide_all_windows();
   set_functions();
   reset_state=T_FALSE;
   wGblQuitFlag=EMU_RESTART;
  }
 }
 parameter->executed=PARAMETER_EXECUTED;
 return parameter->ebene;
}

T_INT vdefault_tasten_funktion(PARAMETER *parameter)
{
 switch (parameter->taste)
 {
  case 0:
   break;
  case KEY_SPACE:
  {
   execute_vtoggle_gui_button(parameter);
   parameter->taste=0;
   break;
  }
  case KEY_ALT_F1:
  case KEY_ALT_F2:
  case KEY_ALT_F3:
  case KEY_ALT_F4:
  case KEY_ALT_F5:
  case KEY_ALT_F6:
  case KEY_ALT_F7:
  case KEY_ALT_F8:
  case KEY_ALT_F9:
  case KEY_ALT_F10:
  case KEY_ALT_F11:
  case KEY_ALT_F12:
  {
   fCarouselSelect(parameter->taste);
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_1:
  case KEY_2:
  case KEY_3:
  case KEY_4:
  case KEY_5:
  case KEY_6:
  case KEY_7:
  case KEY_8:
  case KEY_9:
  case KEY_0:
  {
   fLoadState(parameter->taste);
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_ALT_1:
  case KEY_ALT_2:
  case KEY_ALT_3:
  case KEY_ALT_4:
  case KEY_ALT_5:
  case KEY_ALT_6:
  case KEY_ALT_7:
  case KEY_ALT_8:
  case KEY_ALT_9:
  case KEY_LCURLY:
  case KEY_RCURLY:
  case KEY_LSQRB:
  case KEY_RSQRB:
  {
   fSaveState(parameter->taste);
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_ALT_D:
  {
   fDebugToggle(parameter->taste);
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_ALT_M:
  {
   fDebugMonitorToggle(parameter->taste);
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_ALT_F:
  {
   fDebugFileToggle(parameter->taste);
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_ALT_R:
  {
   fSequencerReset(parameter->taste);
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_ALT_B:
  {
   fSequencerReboot(parameter->taste);
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_ALT_P:
  {
   if (bGblPauseMode==T_TRUE)
   {
    bGblPauseMode=T_FALSE;
    vectrex_update->frequency=emulator_update_frequency;
   }
   else
   {
    bGblPauseMode=T_TRUE;
    vectrex_update->frequency=1;
//    vector_undraw();
//    vector_refresh();
   }
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_ALT_G:
  {
   fScreenshot(parameter->taste);
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_ALT_E:
  {
   fEnableEvent(parameter->taste);
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_F1:
  {
   parameter->executed=PARAMETER_EXECUTED;
   help_function("index.hlf",NULL);
   parameter->taste=0;
   break;
  }
  case KEY_G:
  {
   parameter->executed=PARAMETER_EXECUTED;
   help_function("vgames.hlf",NULL);
   parameter->taste=0;
   break;
  }
  case KEY_ALT_X:
  case KEY_ESCAPE:
  {
   parameter->executed=PARAMETER_EXECUTED;
   execute_vexit_button(parameter);
   parameter->taste=0;
   break;
  }
  case KEY_U:
  {
   parameter->executed=PARAMETER_EXECUTED;
   vector_undraw();
   parameter->taste=0;
   break;
  }
  case KEY_V:
  {
   parameter->executed=PARAMETER_EXECUTED;
   vector_refresh();
   parameter->taste=0;
   break;
  }
/*
  case KEY_C:
  {
   parameter->executed=PARAMETER_EXECUTED;
   execute_vconfig_button(parameter);
   parameter->taste=0;
   break;
  }
*/
  case KEY_L:
  {
   parameter->executed=PARAMETER_EXECUTED;
   execute_vload_button(parameter);
   parameter->taste=0;
   break;
  }
  case KEY_SHIFT_F1:
  {
   fDebugLevelLow(parameter->taste);
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_SHIFT_F2:
  {
   fDebugLevelMid(parameter->taste);
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_SHIFT_F3:
  {
   fDebugLevelHigh(parameter->taste);
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_SHIFT_F8:
  {
   back_show();
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_SHIFT_END:
  {
   dac_offset=0;
   integrator_offset_x=0;
   integrator_offset_y=0;
   wGblXDrift=0;
   wGblYDrift=0;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_MINUS:
  {
   dac_offset++;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_PLUS:
  {
   dac_offset--;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_SHIFT_LEFT:
  {
   integrator_offset_x-=10;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_SHIFT_RIGHT:
  {
   integrator_offset_x+=10;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_SHIFT_UP:
  {
   integrator_offset_y+=10;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_SHIFT_DOWN:
  {
   integrator_offset_y-=10;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_CTRL_LEFT:
  {
   wGblXDrift--;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_CTRL_RIGHT:
  {
   wGblXDrift++;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_CTRL_UP:
  {
   wGblYDrift++;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_CTRL_DOWN:
  {
   wGblYDrift--;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_CTRL_F:
  {
   bGblForceSingleProcessor=(bGblForceSingleProcessor+1)%2;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }

#ifdef ANALOG_CYCLE_PATCH
  case KEY_b:
  {
   BLANK_REACT_OFFSET_CYCLES++;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_B:
  {
   if (BLANK_REACT_OFFSET_CYCLES>1)
   BLANK_REACT_OFFSET_CYCLES--;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_r:
  {
   RAMP_REACT_OFFSET_CYCLES++;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_R:
  {
   if (RAMP_REACT_OFFSET_CYCLES>1)
    RAMP_REACT_OFFSET_CYCLES--;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_h:
  {
   SAMPLE_HOLD_REACT_OFFSET_CYCLES++;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_H:
  {
   if (SAMPLE_HOLD_REACT_OFFSET_CYCLES>1)
   SAMPLE_HOLD_REACT_OFFSET_CYCLES--;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_c:
  {
   DAC_REACT_OFFSET_CYCLES++;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_C:
  {
   if (DAC_REACT_OFFSET_CYCLES>1)
   DAC_REACT_OFFSET_CYCLES--;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_z:
  {
   ZERO_REACT_OFFSET_CYCLES++;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
  case KEY_Z:
  {
   if (ZERO_REACT_OFFSET_CYCLES>1)
   ZERO_REACT_OFFSET_CYCLES--;
   parameter->executed=PARAMETER_EXECUTED;
   parameter->taste=0;
   break;
  }
#endif
  default:
  {
   break;
  }
 }
 return parameter->ebene;
}
